import { Form, FormField, FormLabel, FormItem, FormControl, FormMessage } from "@/components/ui/form";
import { ClipboardCheckIcon, PlusIcon } from "lucide-react";
import { useTranslation } from "react-i18next";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm, UseFormReturn } from "react-hook-form";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AlertDialog, AlertDialogTitle, AlertDialogHeader, AlertDialogContent, AlertDialogTrigger, AlertDialogDescription, AlertDialogFooter, AlertDialogCancel } from "@/components/ui/alert-dialog";
import { useState } from "react";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useGetScheduledScans } from "@/components/nmap/scheduledscans/queries";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Label } from "@/components/ui/label";
import { type scheduledNmapScan } from "@/components/nmap/scheduledscans/types";
import { TFunction } from "i18next";
import { useCreateCheck } from "@/components/alerting/queries";
import { AlertingType } from "../../types";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import CheckReceiverSelector from "../../CheckReceiverSelector";
import { useAuth } from "@/contexts/AuthContext";
import { isAdmin } from "@/components/role/utils";
import { Switch } from "@/components/ui/switch"
import { isThresholdValid, thresholdRegex } from "@/components/alerting/utils";

const nmapCheckFormSchema = z.object({
  check_type: z.literal('nmap'),
  step1: z.object({
    name: z.string().min(1, "Check Name Required").max(255, "Must be less than 255 characters"),
    association: z.object({
      type: z.enum(['scheduled_scan']),
      id: z.string(),
    }).superRefine((data, ctx) => {
      if (!data.id) {
        ctx.addIssue({
          code: "custom",
          message: "Check Association Required",
          path: ["type"],
        });
        ctx.addIssue({
          code: "custom",
          message: "Check Association Required",
          path: ["id"],
        });
      }
    }),
  }),
  step2: z.object({
    metric: z.enum(['ports_open', 'ports_closed']),
    forcecheck: z.boolean().optional(),
    warning_threshold: z
      .string()
      .regex(thresholdRegex, "Invalid threshold syntax")
      .transform(val => val ?? ''),
    critical_threshold: z
      .string()
      .regex(thresholdRegex, "Invalid threshold syntax")
      .transform(val => val ?? ''),
  }).superRefine((data, ctx) => {
    if (!isThresholdValid(data.warning_threshold)){
      ctx.addIssue({
        code: "custom",
        message: "The start of the range must be less than or equal to the end of the range",
        path: ["warning_threshold"],
      });
    }
    if (!isThresholdValid(data.critical_threshold)){
      ctx.addIssue({
        code: "custom",
        message: "The start of the range must be less than or equal to the end of the range",
        path: ["critical_threshold"],
      });
    }
  }),
  step3: z.object({
    user: z.array(z.number()),
    nagios: z.array(z.number()),
    snmp_receiver: z.array(z.number()),
    command: z.array(z.number()),
  }),
})

export default function AddNmapCheckModal() {
  const { t } = useTranslation();
  const { data: nmapScans } = useGetScheduledScans();
  const { user } = useAuth();
  const createCheck = useCreateCheck();
  
  const [newNmapCheckFormOpen, setNewNmapCheckFormOpen] = useState(false);
  const [formStep, setFormStep] = useState<'step1' | 'step2' | 'step3'>('step1');

  const form = useForm({
    resolver: zodResolver(nmapCheckFormSchema),
    mode: 'onTouched',
    defaultValues: {
      check_type: 'nmap',
      step1: {
        name: '',
        association: {
          type: 'scheduled_scan',
          id: '',
        }
      },
      step2: {
        metric: 'ports_open',
        warning_threshold: '',
        critical_threshold: '',
      },
      step3: {
        user: [],
        nagios: [],
        snmp_receiver: [],
        command: [],
      },
    },
  })

  // handle form open
  const handleFormOpen = (open: boolean) => {
    setNewNmapCheckFormOpen(open);
    if (open) {
      setFormStep('step1');
    } else {
      form.reset();
    }
  }

  // handle movement between each step
  const handleStepping = async ({ direction }: { direction: 'next' | 'previous' }) => {
    const steps = ['step1', 'step2', 'step3'];
    const currentStepIndex = steps.indexOf(formStep);

    if (direction === 'next') {
      const isValid = await form.trigger(formStep);
      if (!isValid) return;
    }

    const nextStepIndex = direction === 'next' ? currentStepIndex + 1 : currentStepIndex - 1;
    const nextStep = steps[nextStepIndex];

    if (nextStep) {
      setFormStep(nextStep as 'step1' | 'step2' | 'step3');
    }
  };

  // handle submit
  const handleSubmit = (data: z.infer<typeof nmapCheckFormSchema>) => {
    createCheck.mutate(data);
    handleFormOpen(false);
  };

  // handle cancel
  const handleCancel = () => {
    setNewNmapCheckFormOpen(false);
    setFormStep('step1');
    form.reset();
  };

  return (
    <AlertDialog open={newNmapCheckFormOpen} onOpenChange={handleFormOpen}>
      {isAdmin(user?.role) &&
        <AlertDialogTrigger asChild>
          <Button className="w-fit">
            <PlusIcon />
            {t('New Nmap Check')}
          </Button>
        </AlertDialogTrigger>
      }
      <AlertDialogContent className="flex flex-col max-h-[800px] min-w-[800px] gap-6 overflow-hidden">
        <AlertDialogHeader className="flex flex-col h-min gap-0">
          <AlertingModalTitle formStep={formStep} t={t} />
          <AlertingModalDescription formStep={formStep} t={t} />
        </AlertDialogHeader>
        <Form {...form}>
          {nmapScans && (
            <form onSubmit={form.handleSubmit(handleSubmit)} className="flex flex-col gap-4 grow min-h-0">
              {formStep === 'step1' && (
                <Step1 form={form} nmapScans={nmapScans} t={t} />
              )}
              {formStep === 'step2' && (
                <Step2 form={form} t={t} />
              )}
              {formStep === 'step3' && (
                <Step3 form={form} t={t} />
              )}
            </form>
          )}
        </Form>
        <AlertDialogFooter>
          <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
          {formStep !== 'step1' && (
            <Button type="button" variant="outline" onClick={() => handleStepping({ direction: 'previous' })}>
              {t(`Previous`)}
            </Button>
          )}
          {formStep === 'step1' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step2' && (
            <Button className="w-20" type="button" onClick={() => handleStepping({ direction: 'next' })}>
              {t(`Next`)}
            </Button>
          )}
          {formStep === 'step3' && (
            <Button
              className="w-20"
              onClick={() => handleSubmit(form.getValues())}
              disabled={form.formState.isSubmitting}
            >
              {t(`Create`)}
            </Button>
          )}
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  )
}

const AlertingModalTitle = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3', t: TFunction }) => {
  return (
    <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
      <ClipboardCheckIcon size={20} />
      <span className="w-fit text-lg font-medium">
        {t(`Add Nmap Check`)}
      </span>
      <span className="ml-auto text-base font-medium">
        {
          formStep === 'step1' ? (
            <>
              {t(`Step 1 - Select Scheduled Scan`)}
            </>
          ) : formStep === 'step2' ? (
            <>
              {t(`Step 2 - Select Criteria`)}
            </>
          ) : formStep === 'step3' && (
            <>
              {t(`Step 3 - Select Alerting`)}
            </>
          )
        }
      </span>
    </AlertDialogTitle >
  );
};

const AlertingModalDescription = ({ formStep, t }: { formStep: 'step1' | 'step2' | 'step3', t: TFunction }) => {
  return (
    <AlertDialogDescription>
      {
        formStep === 'step1' ? (
          <>
            {t(`Select the scheduled scan for the check.`)}
          </>
        ) : formStep === 'step2' ? (
          <>
            {t(`Select the criteria you would like to check for.`)}
          </>
        ) : formStep === 'step3' && (
          <>
            {t(`Select how you would like to be notified of this check. Select any of the items in the lists under the tabs, and all the selected elements will be notified.`)}
          </>
        )
      }
    </AlertDialogDescription>
  );
};

const Step1 = ({ form, nmapScans, t }: { form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>, nmapScans: scheduledNmapScan[], t: TFunction }) => {
  const [associationType, setAssociationType] = useState<'scheduled_scan'>('scheduled_scan');

  return (
    <div className="flex flex-col grow gap-8">
      <FormField
        control={form.control}
        name="step1.name"
        render={({ field }) => (
          <FormItem>
            <FormLabel>{t('Name')}</FormLabel>
            <FormControl>
              <Input {...field} />
            </FormControl>
            <FormMessage />
          </FormItem>
        )}
      />

      <div className="flex flex-col gap-0">
        <FormField
          control={form.control}
          name="step1.association.type"
          render={({ field }) => (
            <FormItem className="mb-0 gap-0">
              <FormLabel>{t('Check Association')}</FormLabel>
              <FormControl>
                <RadioGroup
                  defaultValue="scheduled_scan"
                  className="flex items-center gap-8 py-4"
                  onValueChange={(value) => {
                    field.onChange(value);
                    setAssociationType(value as 'scheduled_scan');
                  }}
                >
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem value="scheduled_scan" id="scheduled_scan" />
                    <Label htmlFor="scheduled_scan" className="cursor-pointer">{t('Scheduled Scan')}</Label>
                  </div>
                </RadioGroup>
              </FormControl>
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="step1.association.id"
          render={({ field }) => (
            <FormItem>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => {
                    form.setValue('step1.association.id', value, { shouldValidate: true });
                    form.trigger("step1.association.type")
                  }}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t(`Select a Scheduled Scan`)} />
                  </SelectTrigger>
                  <SelectContent>
                    {
                      associationType === 'scheduled_scan' && nmapScans?.map((scan) => (
                        <SelectItem key={scan.id} value={scan.id?.toString() ?? ''}>
                          {scan.name}
                        </SelectItem>
                      ))
                    }
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  )
}

const Step2 = ({ form, t }: { form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>, t: TFunction }) => {
  return (
    <div className="flex flex-col gap-4 grow min-h-0">
      <div className="flex flex-col flex-row w-full grow gap-8">
        <FormField
          control={form.control}
          name="step2.metric"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Metric')}</FormLabel>
              <FormControl>
                <Select
                  defaultValue={field.value}
                  onValueChange={(value) => field.onChange(value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder={t('Select a metric')} />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ports_open">{t('Ports Open')}</SelectItem>
                    <SelectItem value="ports_closed">{t('Ports Closed')}</SelectItem>
                  </SelectContent>
                </Select>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.forcecheck"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormControl>
                <div className="flex flex-row items-center gap-2">
                    <Switch
                      checked={!!field.value}
                      onCheckedChange={field.onChange}
                    />
                    <FormLabel className="m-0 cursor-pointer select-none">
                      {t('Force Check on Creation')}
                    </FormLabel>
                  </div>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
      <div className="flex w-full flex-col flex-row gap-4">
        <FormField
          control={form.control}
          name="step2.warning_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Warning Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="step2.critical_threshold"
          render={({ field }) => (
            <FormItem className="w-full">
              <FormLabel>{t('Critical Threshold')}</FormLabel>
              <FormControl>
                <Input {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
      </div>
    </div>
  )
}

const Step3 = ({ form, t }: { form: UseFormReturn<z.infer<typeof nmapCheckFormSchema>>, t: TFunction }) => {
  const alertingTypes: AlertingType[] = ['user', 'nagios', 'snmp_receiver', 'command'];

  return (
    <div className="flex flex-col gap-4 grow">
      <Tabs defaultValue={alertingTypes[0]}>
        <TabsList>
          {
            alertingTypes.map((type) => (
              <TabsTrigger key={type} value={type}>{t(type !== 'snmp_receiver' ? type.split('_').map((word) => word.charAt(0).toUpperCase() + word.slice(1)).join(' ') : 'SNMP Receiver')}</TabsTrigger>
            ))
          }
        </TabsList>

        {
          alertingTypes.map((type) => (
            <TabsContent key={type} value={type}>
              <CheckReceiverSelector alertingType={type} form={form} step="step3" t={t} />
            </TabsContent>
          ))
        }

      </Tabs>
    </div>
  );
};