import { Button } from '@/components/ui/button'
import { ActivityIcon, PlusIcon } from 'lucide-react'
import { useState } from 'react'
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog'
import { useCreateServiceHostname, useGetNagiosServers } from '@/components/alerting/queries'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { NagiosServer, ServiceHostnameFormData } from '@/components/alerting/types'
import { z } from 'zod'
import { useForm } from 'react-hook-form'
import { zodResolver } from '@hookform/resolvers/zod'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Input } from '@/components/ui/input'
import { useTranslation } from 'react-i18next'
import { useAuth } from '@/contexts/AuthContext'
import { isAdmin } from '@/components/role/utils'

const alertingServiceHostnameFormSchema = z.object({
  hostname: z.string(),
  servicename: z.string(),
  server_id: z.number(),
})

export default function AddServiceHostnamesModal() {
  const { t } = useTranslation();
  const [newServiceHostnamesFormOpen, setNewServiceHostnamesFormOpen] = useState(false)
  const { data: nagios_servers } = useGetNagiosServers();
  const { mutate: createServiceHostname } = useCreateServiceHostname();
  const { user } = useAuth();

  const form = useForm({
    resolver: zodResolver(alertingServiceHostnameFormSchema),
    mode: 'onBlur', 
    defaultValues: {
      hostname: '',
      servicename: '',
      server_id: parseInt(''),
    }
  })

  const handleSubmit = (data: ServiceHostnameFormData) => {
    createServiceHostname(data)
    form.reset()
  }

  const handleFormOpen = (open: boolean) => {
    setNewServiceHostnamesFormOpen(open)
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={newServiceHostnamesFormOpen} onOpenChange={handleFormOpen}>
      {isAdmin(user?.role) &&
        <AlertDialogTrigger asChild>
          <Button className="w-fit text-xs font-bold">
            <PlusIcon />
            {t(`New Service/Hostname`)}
          </Button>
        </AlertDialogTrigger>
      }
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <ActivityIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Add Service Hostname`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Add a new service hostname to NNA, allowing you to send alerts directly to a service/host on your Nagios server.`)}
        </AlertDialogDescription>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className='flex flex-col gap-4'>
            <FormField
              control={form.control}
              name="servicename"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Servicename`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="hostname"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Hostname`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="server_id"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Nagios Server`)}</FormLabel>
                  <FormControl>
                    <Select
                      defaultValue={field.value.toString()}
                      onValueChange={(value) => {
                        form.setValue('server_id', parseInt(value), { shouldValidate: true });
                      }}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder={t(`Select a Nagios server`)} />
                      </SelectTrigger>
                      <SelectContent>
                        {
                          nagios_servers?.map((server: NagiosServer) => (
                            <SelectItem key={server.id} value={server.id?.toString() ?? ''}>
                              {server.name}
                            </SelectItem>
                          ))
                        }
                      </SelectContent>
                    </Select>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel type='button' onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <AlertDialogAction disabled={!form.formState.isValid || form.formState.isSubmitting} type="submit" className="w-fit">
                {t(`Add Service/Hostname`)}
              </AlertDialogAction>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  )
}
