import { AlertDialog, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog'
import { z } from 'zod'
import { zodResolver } from '@hookform/resolvers/zod'
import { useForm } from 'react-hook-form'
import { useState } from 'react'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from '@/components/ui/form'
import { Input } from '@/components/ui/input'
import { Checkbox } from '@/components/ui/checkbox'
import { NagiosServerFormData } from '@/components/alerting/types'
import { useUpdateNagiosServer } from '@/components/alerting/queries'
import { NagiosServer } from '@/components/alerting/types'
import { DropdownMenuItem } from '@/components/ui/dropdown-menu'
import { HardDriveIcon, PencilIcon } from 'lucide-react'
import { Button } from '@/components/ui/button'
import { useTranslation } from 'react-i18next'

const nagiosServerFormSchema = z.object({
  name: z.string().min(1),
  nrdp_url: z.string().url(),
  nrdp_token: z.string().min(1),
  verify_ssl: z.boolean().default(true),
})

export default function EditNagiosServerModal({ nagiosServer, onModalClose }: { nagiosServer: NagiosServer, onModalClose: () => void }) {
  const { t } = useTranslation();
  const [editNagiosServerFormOpen, setEditNagiosServerFormOpen] = useState(false)
  const { mutate: updateNagiosServer } = useUpdateNagiosServer();

  const form = useForm({
    resolver: zodResolver(nagiosServerFormSchema),
    mode: 'onChange',
    defaultValues: {
      name: nagiosServer.name || '',
      nrdp_url: nagiosServer.nrdp_url || '',
      nrdp_token: nagiosServer.nrdp_token || '',
      verify_ssl: nagiosServer.verify_ssl ?? true,
    },
  })

  const handleSubmit = (data: NagiosServerFormData) => {
    updateNagiosServer({ id: nagiosServer.id, data })
    handleFormOpen(false)
  }

  const handleFormOpen = (open: boolean) => {
    setEditNagiosServerFormOpen(open)
    if (open) {
      form.reset()
    }
    if (!open) {
      onModalClose()
    }
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={editNagiosServerFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <HardDriveIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Edit Nagios Server`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Edit the Nagios server settings.`)}
        </AlertDialogDescription>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className="flex flex-col gap-4 grow min-h-0">
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Name`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_url"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP URL`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="nrdp_token"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`NRDP Token`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            
            <FormField
              control={form.control}
              name="verify_ssl"
              render={({ field }) => (
                <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                  <FormControl>
                    <Checkbox
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                  </FormControl>
                  <div className="space-y-1 leading-none">
                    <FormLabel>
                      {t(`Verify SSL Certificate`)}
                    </FormLabel>
                    <FormDescription>
                      {t(`Disable this only for self-signed certificates or trusted internal servers if NRDP is serving over a secure connection.`)}
                    </FormDescription>
                  </div>
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <Button
                type="submit"
                disabled={!form.formState.isValid || form.formState.isSubmitting}
                className="w-fit"
              >
                {t(`Update Nagios Server`)}
              </Button>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  )
}
