import { ActivityIcon, PencilIcon } from 'lucide-react'
import { useState } from 'react'
import { AlertDialog, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog'
import { useGetNagiosServers, useUpdateServiceHostname } from '@/components/alerting/queries'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { NagiosServer, ServiceHostname, ServiceHostnameFormData } from '@/components/alerting/types'
import { z } from 'zod'
import { useForm } from 'react-hook-form'
import { zodResolver } from '@hookform/resolvers/zod'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Input } from '@/components/ui/input'
import { DropdownMenuItem } from '@/components/ui/dropdown-menu'
import { Button } from '@/components/ui/button'
import { useTranslation } from 'react-i18next'

const alertingServiceHostnameFormSchema = z.object({
  hostname: z.string(),
  servicename: z.string(),
  server_id: z.number(),
})

  export default function EditServiceHostnameModal({ serviceHostname, onModalClose }: { serviceHostname: ServiceHostname, onModalClose: () => void }) {
  const { t } = useTranslation();
  const [editServiceHostnameFormOpen, setEditServiceHostnameFormOpen] = useState(false)
  const { data: nagios_servers } = useGetNagiosServers();
  const { mutate: updateServiceHostname } = useUpdateServiceHostname();

  const form = useForm({
    resolver: zodResolver(alertingServiceHostnameFormSchema),
    mode: 'onBlur',
    defaultValues: {
      hostname: serviceHostname.hostname || '',
      servicename: serviceHostname.servicename || '',
      server_id: serviceHostname.server_id || parseInt(''),
    }
  })

  const handleSubmit = (data: ServiceHostnameFormData) => {
    updateServiceHostname({ id: serviceHostname.id, data })
    handleFormOpen(false)
  }

  const handleFormOpen = (open: boolean) => {
    setEditServiceHostnameFormOpen(open)
    if (open) {
      form.reset()
    }
    if (!open) {
      onModalClose()
    }
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={editServiceHostnameFormOpen} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <ActivityIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Edit Service Hostname`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Edit the service hostname settings.`)}
        </AlertDialogDescription>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleSubmit)} className='flex flex-col gap-4'>
            <FormField
              control={form.control}
              name="servicename"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Servicename`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="hostname"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Hostname`)}</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="server_id"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t(`Nagios Server`)}</FormLabel>
                  <FormControl>
                    <Select
                      defaultValue={field.value.toString()}
                      onValueChange={(value) => {
                        form.setValue('server_id', parseInt(value), { shouldValidate: true });
                      }}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder={t(`Select a Nagios server`)} />
                      </SelectTrigger>
                      <SelectContent>
                        {
                          nagios_servers?.map((server: NagiosServer) => (
                            <SelectItem key={server.id} value={server.id?.toString() ?? ''}>
                              {server.name}
                            </SelectItem>
                          ))
                        }
                      </SelectContent>
                    </Select>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel type='button' onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <Button
                type="submit"
                disabled={!form.formState.isValid || form.formState.isSubmitting}
                className="w-fit"
              >
                {t(`Update Service/Hostname`)}
              </Button>
            </AlertDialogFooter>
          </form>
        </Form>
      </AlertDialogContent>
    </AlertDialog>
  )
}
