import { PencilIcon, TargetIcon } from 'lucide-react'
import { useState } from 'react'
import { AlertDialog, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '@/components/ui/alert-dialog'
import { z } from "zod"
import { useForm } from "react-hook-form"
import { FormField, FormItem, FormLabel, FormControl, FormMessage, Form } from "@/components/ui/form"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { zodResolver } from '@hookform/resolvers/zod'
import { SNMPReceiver, SNMPReceiverFormData } from '@/components/alerting/types'
import { useUpdateSNMPReceiver } from '@/components/alerting/queries'
import { DropdownMenuItem } from '@/components/ui/dropdown-menu'
import { Button } from '@/components/ui/button'
import { useTranslation } from 'react-i18next'

const snmp_receiver_form_schema = z.object({
  name: z.string().min(1, "SNMP receiver name required").max(255, "Must be less than 255 characters"),
  version: z.enum(['2c', '3']).default('2c'),
  community_string: z.string().optional(),
  auth_level: z.enum(['AuthPriv', 'NoAuthNoPriv', 'AuthNoPriv']).default('AuthPriv'),
  priv_protocol: z.string().optional(),
  priv_password: z.string().optional(),
  auth_protocol: z.string().optional(),
  auth_password: z.string().optional(),
  username: z.string().optional(),
  port: z.coerce.number<number>().min(1, "Port must be between 1 and 65535").max(65535, "Port must be between 1 and 65535"),
  ip: z.string().min(1, "IP address required").regex(/^(?:(?:25[0-5]|2[0-4]\d|1?\d{1,2})(?:\.(?!$)|$)){4}$/, "Must be a valid IP address"),
}).superRefine((data, ctx) => {
  if (data.version === '3' && !data.auth_level) {
    ctx.addIssue({
      code: "custom",
      message: "Auth level is required for SNMP v3",
      path: ["auth_level"],
    })
  }
  if (data.version === '3' && !data.auth_protocol) {
    ctx.addIssue({
      code: "custom",
      message: "Auth protocol is required for SNMP v3",
      path: ["auth_protocol"],
    })
  }
  if (data.version === '3' && !data.auth_password) {
    ctx.addIssue({
      code: "custom",
      message: "Auth password is required for SNMP v3",
      path: ["auth_password"],
    })
  }
  if (data.version === '3' && !data.username) {
    if (data.auth_level === 'AuthPriv' || data.auth_level === 'AuthNoPriv') {
      ctx.addIssue({
        code: "custom",
        message: "Username is required for SNMP v3",
        path: ["username"],
      })
    }
  }
})

export default function EditSNMPReceiverModal({ snmp_receiver, onModalClose }: { snmp_receiver: SNMPReceiver, onModalClose: () => void }) {
  const { t } = useTranslation();
  const { mutate: update_snmp_receiver } = useUpdateSNMPReceiver();
  const [edit_snmp_receiver_form_open, set_edit_snmp_receiver_form_open] = useState(false)

  const form = useForm({
    resolver: zodResolver(snmp_receiver_form_schema),
    mode: 'onBlur',
    defaultValues: {
      name: snmp_receiver.name || '',
      version: snmp_receiver.version || '2c',
      community_string: snmp_receiver.community_string || '',
      auth_level: snmp_receiver.auth_level || 'AuthPriv',
      priv_protocol: snmp_receiver.priv_protocol || '',
      priv_password: snmp_receiver.priv_password || '',
      auth_protocol: snmp_receiver.auth_protocol || '',
      auth_password: snmp_receiver.auth_password || '',
      username: snmp_receiver.username || '',
      port: snmp_receiver.port || 162,
      ip: snmp_receiver.ip || '',
    },
  });

  const onSubmit = (data: SNMPReceiverFormData) => {
    update_snmp_receiver({ id: snmp_receiver.id, data })
    handleFormOpen(false)
  }

  const handleFormOpen = (open: boolean) => {
    set_edit_snmp_receiver_form_open(open)
    if (open) {
      form.reset()
    }
    if (!open) {
      onModalClose()
    }
  }

  const handleCancel = () => {
    handleFormOpen(false)
    form.reset()
  }

  return (
    <AlertDialog open={edit_snmp_receiver_form_open} onOpenChange={handleFormOpen}>
      <AlertDialogTrigger asChild>
        <DropdownMenuItem onSelect={(e) => e.preventDefault()}>
          <PencilIcon />
          <span className="hidden lg:block">{t(`View/Edit`)}</span>
        </DropdownMenuItem>
      </AlertDialogTrigger>
      <AlertDialogContent className='flex flex-col min-w-[800px]'>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex h-min w-full mb-2 items-center gap-2">
            <TargetIcon size={20} />
            <span className="text-lg font-medium">
              {t(`Edit SNMP Receiver`)}
            </span>
          </AlertDialogTitle>
        </AlertDialogHeader>
        <AlertDialogDescription>
          {t(`Edit an existing SNMP receiver to be used in your checks.`)}
        </AlertDialogDescription>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="flex flex-col gap-4 w-full">

            <div className='flex flex-row gap-4 w-full'>

              <div className="flex flex-col gap-4 w-full">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t(`SNMP Receiver Name`)}</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder={t("Enter SNMP receiver name")} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="ip"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t(`IP Address`)}</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder={t("IP address (e.g. 192.168.1.100)")} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="port"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t(`Port`)}</FormLabel>
                      <FormControl>
                        <Input {...field} placeholder={t("Port number (e.g. 162)")} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="version"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>{t(`SNMP Version`)}</FormLabel>
                      <Select onValueChange={field.onChange} defaultValue={field.value?.toString()}>
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder={t("Select SNMP version")} />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="2c">{`2c`}</SelectItem>
                          <SelectItem value="3">{`3`}</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="flex flex-col gap-4 w-full">
                {
                  form.watch('version') === '2c' ? (
                    <>
                      <FormField
                        control={form.control}
                        name="community_string"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>{t(`Community String`)}</FormLabel>
                            <FormControl>
                              <Input {...field} placeholder={t("Community string (e.g. public)")} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </>
                  ) : (
                    <>
                      <FormField
                        control={form.control}
                        name="username"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>{t(`Username`)}</FormLabel>
                            <FormControl>
                              <Input {...field} placeholder={t("Username (for SNMP v3)")} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                      <FormField
                        control={form.control}
                        name="auth_level"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>{t(`Auth Level`)}</FormLabel>
                            <FormControl>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger>
                                    <SelectValue placeholder={t("Select auth level")} />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  <SelectItem value="AuthPriv">{`AuthPriv`}</SelectItem>
                                  <SelectItem value="NoAuthNoPriv">{`NoAuthNoPriv`}</SelectItem>
                                  <SelectItem value="AuthNoPriv">{`AuthNoPriv`}</SelectItem>
                                </SelectContent>
                              </Select>
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                      {
                        (form.watch('auth_level') !== 'NoAuthNoPriv') && (
                          <>
                            <FormField
                              control={form.control}
                              name="auth_protocol"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>{t(`Auth Protocol`)}</FormLabel>
                                  <Select onValueChange={field.onChange} defaultValue={field.value}>
                                    <FormControl>
                                      <SelectTrigger>
                                        <SelectValue placeholder={t("Select auth protocol")} />
                                      </SelectTrigger>
                                    </FormControl>
                                    <SelectContent>
                                      <SelectItem value="MD5">{`MD5`}</SelectItem>
                                      <SelectItem value="SHA">{`SHA`}</SelectItem>
                                    </SelectContent>
                                  </Select>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={form.control}
                              name="auth_password"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>{t(`Auth Password`)}</FormLabel>
                                  <FormControl>
                                    <Input {...field} type="password" placeholder={t("Auth password")} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                          </>
                        )
                      }
                      {
                        (form.watch('auth_level') === 'AuthPriv') && (
                          <>
                            <FormField
                              control={form.control}
                              name="priv_protocol"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>{t(`Privacy Protocol`)}</FormLabel>
                                  <Select onValueChange={field.onChange} defaultValue={field.value?.toString()}>
                                    <FormControl>
                                      <SelectTrigger>
                                        <SelectValue placeholder={t("Select privacy protocol")} />
                                      </SelectTrigger>
                                    </FormControl>
                                    <SelectContent>
                                      <SelectItem value="AES">{`AES`}</SelectItem>
                                      <SelectItem value="DES">{`DES`}</SelectItem>
                                    </SelectContent>
                                  </Select>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                            <FormField
                              control={form.control}
                              name="priv_password"
                              render={({ field }) => (
                                <FormItem>
                                  <FormLabel>{t(`Privacy Password`)}</FormLabel>
                                  <FormControl>
                                    <Input {...field} type="password" placeholder={t("Privacy password")} />
                                  </FormControl>
                                  <FormMessage />
                                </FormItem>
                              )}
                            />
                          </>
                        )
                      }
                    </>
                  )
                }
              </div>
            </div>
            {/* FOOTER */}
            <AlertDialogFooter>
              <AlertDialogCancel type='button' onClick={handleCancel}>{t(`Cancel`)}</AlertDialogCancel>
              <Button type="submit" disabled={!form.formState.isValid || form.formState.isSubmitting || !form.formState.isDirty} className="w-fit">
                {t(`Update SNMP Receiver`)}
              </Button>
            </AlertDialogFooter>
          </form>
        </Form>

      </AlertDialogContent>
    </AlertDialog>
  )
}
