"use client"

import { useTranslation } from "react-i18next";
import { Table as Icon } from "lucide-react";
import { DashletDataOptions } from "@/components/dashboard/types";
import { useGetSourceTopTalkersData } from "@/components/dashboard/dashlets/queries";
import { DataTable as Table } from "@/components/ui/DataTable/DataTable";
import { ColumnDef } from "@tanstack/react-table";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";

export const meta = {
  name: 'table',
  title: 'Data Table',
  description: 'A simple data table',
  icon: Icon,
};

const defaultTalkerOptions = {
  source_id: 0,
  num_talkers: [5],
  order_by: "bytes",
  group_by: "srcip",
}

type DataTableData = {
  start: string,
  end: string,
  duration: string,
  bpp: string,
  bps: string,
  bytes: string,
  bytespercent: string,
  flows: string,
  flowspercent: string,
  packets: string,
  packetspercent: string,
  pps: string,
  protocol: string,
}

const defaultColumnVisibility = {
  bpp: false,
  bps: false,
  bytespercent: false,
  flowspercent: false,
  packetspercent: false,
  pps: false,
  start: false,
  end: false,
}

export default function DataTable(props: DashletDataOptions) {

  const { t } = useTranslation();
  const {
    data: talkersData,
    isLoading: talkersLoading,
    isError: talkersError,
  } = useGetSourceTopTalkersData(Number(props.top_talker_data_options?.source_id) ?? 0, props.range, props.top_talker_data_options ?? defaultTalkerOptions, {
    enabled: props.type === "talkers",
  });

  const data = props.type === "talkers" ? talkersData : {};
  const isLoading = props.type === "talkers" ? talkersLoading : false;
  const isError = props.type === "talkers" ? talkersError : false;
  const nameKey = props.type === "talkers" ? props.top_talker_data_options?.group_by ?? "srcip" : "srcip";

  const groups: Record<string, string> = {
    srcip: t("Source IP"),
    dstip: t("Destination IP"),
    srcport: t("Source Port"),
    dstport: t("Destination Port"),
  }

  const columns: ColumnDef<DataTableData>[] = [
    {
      accessorKey: 'start',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Start")} />,
    },
    {
      accessorKey: 'end',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("End")} />,
    },
    {
      accessorKey: 'duration',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Duration")} />,
    },
    {
      accessorKey: 'protocol',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Protocol")} />,
    },
    {
      accessorKey: `${nameKey}`,
      header: ({ column }) => <DataTableColumnHeader column={column} title={groups[`${nameKey}`]} />,
    },
    {
      accessorKey: 'bpp',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Bits/Packet")} />,
    },
    {
      accessorKey: 'bps',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Bits/Second")} />,
    },
    {
      accessorKey: 'bytes',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Bytes")} />,
    },
    {
      accessorKey: 'bytespercent',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Bytes Percent")} />,
    },
    {
      accessorKey: 'flows',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Flows")} />,
    },
    {
      accessorKey: 'flowspercent',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Flows Percent")} />,
    },
    {
      accessorKey: 'packets',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Packets")} />,
    },
    {
      accessorKey: 'packetspercent',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Packets Percent")} />,
    },
    {
      accessorKey: 'pps',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Packets/Second")} />,
    },
  ];

  if (isLoading || !data) return <DataTableSkeleton />;
  if (isError) return <div>{t("Data not found")}</div>

  return (
    <Table hidePageSizeSelector hidePagination columns={columns} data={data[nameKey]?.slice(1)} defaultColumnVisibility={defaultColumnVisibility} />
  );
}
