import { dashboard, dashlet } from "@/components/dashboard/types";
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { useNavigate, useParams } from "react-router-dom";
import { toast } from "sonner";

export function useGetDashboard(id: number) {
  return useQuery<dashboard, AxiosError>({
    queryKey: ['dashboard', id],
      queryFn: async () => {
          const response = await axios.get('/api/v1/dashboards/' + id);
          return response.data;
      },
      retry: false,
  });
}

export function useGetDashboards() {
  return useQuery<dashboard[], AxiosError>({
      queryKey: ['dashboards'],
      queryFn: async () => {
          const response = await axios.get('/api/v1/dashboards');
          return response.data;
      },
  });
}

export function useAddDashboard() {
  const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (data: Omit<dashboard, "user_id"> & { user_id?: number }) => {
            return axios.post(
                '/api/v1/dashboards',
                {
                    id: data.id,
                    user_id: data.user_id,
                    title: data.title,
                    range: data.range,
                    refresh_rate: data.refresh_rate,
                    icon: data.icon,
                },
                {
                    headers: {
                        'content-type': 'application/x-www-form-urlencoded',
                    },
                }
            );
        },
        onSuccess: () => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            toast(t('Dashboard added.'),
              { description: t('Access your new dashboard via My Dashboards.') });
        },
        onError: () => {
            toast(t('Failed to add dashboard.'),
              { description: t('Please try again.') });
        },
    });
}

export function usePatchDashboard() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: (dashboard: dashboard) => {
            return axios.patch(
                `/api/v1/dashboards/${dashboard.id}`,
                {
                    title: dashboard.title,
                    range: dashboard.range,
                    refresh_rate: dashboard.refresh_rate,
                    icon: dashboard.icon,
                },
            );
        },
        onSuccess: (_response, variables) => {
            queryClient.invalidateQueries({ queryKey: ['dashboard', variables.id] });
        },
        onError: () => {
            toast(t('Failed to update dashboard.'),
              { description: t('Please try again.') });
        },
    });
}

export function useEditDashboard() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();

    return useMutation({
        mutationFn: ({ id, dashlets }: { id: number, dashlets: dashlet[]; }) => {
            return axios.put(
                `/api/v1/dashboards/${id}`,
                {
                    dashlets,
                },
            );
        },
        onSuccess: (_response, variables) => {
            queryClient.invalidateQueries({ queryKey: ['dashlets', variables.id] });
        },
        onError: () => {
            toast(t('Failed to update dashboard.'),
              { description: t('Please try again.') });
        },
    });
}

export function useDeleteDashboard() {
    const { t } = useTranslation();
    const queryClient = useQueryClient();
    const { dashboardId } = useParams();
    const redirect = useNavigate();

    return useMutation({
        mutationFn: (id: number) => {
            return axios.delete(`/api/v1/dashboards/${id}`);
        },
        onSuccess: (_reponse, id) => {
            queryClient.invalidateQueries({ queryKey: ['dashboards'] });
            if (`${id}` === dashboardId) {
                const data: dashboard[] | undefined = queryClient.getQueryData(['dashboards']);
                if (data && data.length > 1) {
                    redirect(`/dashboards/${data[data.length - 2].id}`);
                } else {
                    redirect('/dashboards');
                }
            }
            toast(t('Dashboard deleted'));
        },
        onError: () => {
            toast(t('Failed to delete dashboard'),
              { description: t('Please try again') });
        },
    });
}