import DashboardDropdown from "@/components/dashboard/DashboardDropdown";
import TimeDropdown from "@/components/dashboard/toolbar/TimeDropdown";
import { AddDashletDialog } from "@/components/dashboard/dashlet/AddEditDashletDialog";
import { Button } from "@/components/ui/button";
import { Plus, ChevronsUpDown, ClipboardPlus, Pencil, Check } from "lucide-react";
import { useDashboard } from "@/components/dashboard/DashboardContext";
import ExportHoverCard from "@/components/dashboard/toolbar/ExportHoverCard";
import { AddReportDialog } from "@/components/report/AddEditReportDialog"
import { useTranslation } from "react-i18next";
import { useEffect, useLayoutEffect, useRef, useState } from "react";
import { cn } from "@/lib/utils";
import { usePatchDashboard } from "@/components/dashboard/queries";
import { useQueryClient } from "@tanstack/react-query";
import { dashboard } from "@/components/dashboard/types";
import { Separator } from "@/components/ui/separator";
import DefaultDashboardButton from "@/components/dashboard/toolbar/DefaultDashboardButton";

function DashboardToolbarTitle(dashboard: dashboard) {
  const inputRef = useRef<HTMLInputElement>(null);
  const spanRef = useRef<HTMLDivElement>(null);
  const wrapperRef = useRef<HTMLDivElement>(null);
  const invisibleRef = useRef<HTMLSpanElement>(null);
  const [isEditing, setIsEditing] = useState(false);
  const [value, setValue] = useState(dashboard.title);
  const [width, setWidth] = useState(0);
  const patchDashboardMutation = usePatchDashboard();
  const queryClient = useQueryClient();

  useEffect(() => {
    if (!isEditing) return;
    const handleClickOutside = (e: MouseEvent) => {
      if (isEditing && wrapperRef.current && !wrapperRef.current.contains(e.target as Node)) {
        patchDashboardMutation.mutate({...dashboard, title: value}, { onSuccess: () => queryClient.invalidateQueries({ queryKey: ['dashboards'] }) });
        setIsEditing(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [isEditing]);

  useLayoutEffect(() => {
    invisibleRef.current && setWidth(invisibleRef.current.getBoundingClientRect().width);
  }, [value]);
  
  const handleEdit = () => {
    setIsEditing(true);
    inputRef.current?.select();
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setValue(e.target.value);
  }

  const handleKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === "Enter") {
      patchDashboardMutation.mutate({...dashboard, title: value}, { onSuccess: () => queryClient.invalidateQueries({ queryKey: ['dashboards'] }) });
      inputRef.current?.blur();
      setIsEditing(false);
    }
  };

  return (
    <>
      <div ref={wrapperRef} className="flex">
        <input
          ref={inputRef}
          className="text-xl font-semibold outline-none dark:outline-none px-0"
          maxLength={50}
          value={value}
          onChange={(e) => handleChange(e)}
          onKeyDown={(e) => handleKeyDown(e)}
          style={{ width: isEditing ? width : 0 }}
        />
        <div
          ref={spanRef}
          className={cn(isEditing && "hidden", "max-w-100 font-semibold truncate text-xl")}
        >
          {value}
        </div>
      </div>
      <Button
        key={`${isEditing}`}
        variant="ghost"
        className="size-8 opacity-0 group-hover/toolbar:opacity-100 transition-[color,opacity]"
        {...(!isEditing && { onClick: handleEdit })}
      >
        <Check
          className={`absolute transition-all duration-200 animate-in fade-in zoom-in ${
            isEditing ? "scale-100" : "scale-0"
          }`}
        />
        <Pencil
          className={`absolute transition-all duration-200 animate-in fade-in zoom-in ${
            isEditing ? "scale-0" : "scale-100"
          }`}
        />
      </Button>
      <span ref={invisibleRef} className="absolute invisible text-xl font-semibold whitespace-pre px-0">{value}</span>
    </>
  )
}

export default function DashboardToolbar() {

  const { t } = useTranslation();
  const { dashboard } = useDashboard();
  
  return (
    <div className="flex justify-between ml-2 mr-3 pt-2">
      <div className="flex gap-2 items-center group/toolbar">
        <DashboardDropdown align="start">
          <Button
            className="w-7 h-8"
            variant="ghost"
            role="combobox"
          >
            <ChevronsUpDown className="text-muted-foreground" />
          </Button>
        </DashboardDropdown>
        <Separator orientation="vertical" className="data-[orientation=vertical]:h-4" />
        <DefaultDashboardButton dashboardId={Number(dashboard.id)} />
        <DashboardToolbarTitle key={dashboard.id} {...dashboard} />
      </div>
      <div className="flex items-center gap-1">
        <AddReportDialog dashboardId={Number(dashboard.id)}>
          <Button size="sm" variant="ghost">
            <ClipboardPlus />
            {t("Schedule")}
          </Button>
        </AddReportDialog>
        <ExportHoverCard dashboardId={Number(dashboard.id)} />
        <TimeDropdown />
        <AddDashletDialog dashboardId={Number(dashboard.id)}>
          <Button size="sm">
            <Plus />{t("Add Dashlet")}
          </Button>
        </AddDashletDialog>
      </div>
    </div>
  )
}