import { useMutation, useQueryClient } from "@tanstack/react-query";
import { useAuth } from "@/contexts/AuthContext";
import axios from "axios";
import { useTranslation } from "react-i18next";
import { Button } from "@/components/ui/button";
import { LayoutDashboard, Home } from "lucide-react";
import { cn } from "@/lib/utils";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip"

export default function DefaultDashboardButton({ dashboardId }: { dashboardId: number }) {
  const { user } = useAuth();
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  const isDefault = user?.default_dashboard === dashboardId;

  const { mutate } = useMutation({
    mutationFn: () => axios.patch(`/api/v1/profile/default-dashboard/${dashboardId}`),
    onSuccess: async () => {
      queryClient.invalidateQueries({ queryKey: ["auth"] });
    },
  });

  return (
    <Tooltip>
      <TooltipTrigger asChild>
        <Button
          aria-label={t("Toggle default dashboard")}
          variant="ghost"
          onClick={() => mutate()}
          className="size-8 perspective-midrange"
        >
          <span
            className={cn(
              "relative transition-transform duration-500 transform-3d",
              isDefault ? "rotate-y-180" : ""
            )}
          >
            <span className="absolute inset-0 flex items-center justify-center backface-hidden translate-z-[1px]">
              <LayoutDashboard />
            </span>
            <span className="absolute inset-0 flex items-center justify-center backface-hidden rotate-y-180">
              <Home />
            </span>
          </span>
        </Button>
      </TooltipTrigger>
      <TooltipContent>
        <p>
          {isDefault
            ? t("Your default dashboard!")
            : t("Set default dashboard")}
        </p>
      </TooltipContent>
    </Tooltip>
  );
}
