import { useTranslation } from 'react-i18next';
import { Button } from "@/components/ui/button";
import { RefreshCw, ChevronDown, Calendar } from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuCheckboxItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from '@/components/ui/tooltip';
import { useEffect, useState } from 'react';
import { cn } from '@/lib/utils';
import { useDashboard } from "@/components/dashboard/DashboardContext";
import { useQueryClient } from "@tanstack/react-query";
import { usePatchDashboard } from '@/components/dashboard/queries';
import { Separator } from "@/components/ui/separator";

export default function TimeDropdown() {

const { t } = useTranslation();

  const queryClient = useQueryClient();
  const patchDashboardMutation = usePatchDashboard();
  const { dashboard } = useDashboard();
  const refreshRate = dashboard.refresh_rate;
  const range = dashboard.range;
  const [rotate, setRotate] = useState(false);

  useEffect(() => {
    if (refreshRate === 0) return;
    const interval = setInterval(() => {
      queryClient.invalidateQueries({ queryKey: ['dashlet'] });
    }, refreshRate);
    return () => clearInterval(interval);
  }, [refreshRate, queryClient]);

  const refreshRateOptions = [
    { label: 'Off', value: 0 },
    { label: '1m', value: 60000 },
    { label: '5m', value: 300000 },
    { label: '15m', value: 900000 },
    { label: '30m', value: 1800000 },
    { label: '1h', value: 3600000 },
    { label: '2h', value: 7200000 },
    { label: '1d', value: 86400000 }
  ];

  const rangeOptions: Record<string, string> = {
    '-15 minute': t('Last 15m'),
    '-1 hour': t('Last 1h'),
    '-2 hour': t('Last 2h'),
    '-6 hour': t('Last 6h'),
    '-12 hour':  t('Last 12h'),
    '-24 hour': t('Last 24h'),
    '-2 day': t('Last 2d'),
    '-7 day': t('Last 7d'),
    '-30 day': t('Last 30d'),
  }

  const refreshNow = () => {
    setRotate(true);
    queryClient.invalidateQueries({ queryKey: ['dashlet'] });
  }

  return (
    <div className="flex items-center justify-between">
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="outline" size="sm" className="flex items-center justify-between gap-2 mr-1">
            <Calendar  />
            {rangeOptions[range]}
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent className="min-w-fit">
          {Object.entries(rangeOptions).map(([key, value]) => (
            <DropdownMenuCheckboxItem
              key={key}
              checked={range === key}
              onClick={() => patchDashboardMutation.mutate({...dashboard, range: key})}
            >
              {value}
            </DropdownMenuCheckboxItem>
          ))}
        </DropdownMenuContent>
      </DropdownMenu>
      <Tooltip>
        <TooltipTrigger asChild>
          <Button
            size="sm"
            variant="ghost"
            onClick={() => refreshNow()}
          >
            <RefreshCw
              fontSize="small"
              className={cn(rotate && "animate-[spin_350ms_linear_1]")}
              onAnimationEnd={() => setRotate(false)}
            />
          </Button>
        </TooltipTrigger>
        <TooltipContent>
          {t("Refresh")}
        </TooltipContent>
      </Tooltip>
      <Separator orientation="vertical" className="data-[orientation=vertical]:h-4" />
      <DropdownMenu>
        <Tooltip>
          <TooltipTrigger asChild>
            <DropdownMenuTrigger asChild>
              <Button size="sm" variant="ghost">
                <ChevronDown />
              </Button>
            </DropdownMenuTrigger>
          </TooltipTrigger>
          <TooltipContent>
            {t("Auto refresh")}
          </TooltipContent>
        </Tooltip>
        <DropdownMenuContent className="min-w-fit">
          {refreshRateOptions.map((option) => (
            <DropdownMenuCheckboxItem
              key={option.value}
              checked={refreshRate === option.value}
              onClick={() => patchDashboardMutation.mutate({...dashboard, refresh_rate: option.value})}
            >
              {option.label}
            </DropdownMenuCheckboxItem>
          ))}
        </DropdownMenuContent>
      </DropdownMenu>
    </div>
  )
}