import {
  Popover,
  PopoverTrigger,
  PopoverContent
} from "@/components/ui/popover";
import { Button } from "@/components/ui/button";
import { useTranslation } from "react-i18next";
import {
  useGetSystemCPU,
  useGetSystemMemory,
  useGetSystemDisk
} from "@/components/header/queries";
import { CPUStatus, MemoryStatus, DiskStatus } from "@/components/header/types";
import { UseQueryResult } from "@tanstack/react-query";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";

type StatusType = 'success' | 'warning' | 'error' | 'unknown' | 'pending';

const StatusColors = {
  success: 'bg-success',
  warning: 'bg-warning',
  error: 'bg-error',
  unknown: 'bg-unknown',
  pending: 'bg-pending',
};

function StatusIndicator({
  status,
}: {
  status: StatusType;
}) {
  const color = StatusColors[status];
  
  return (
    <span className="relative flex h-2 w-2">
      <span className={`absolute inline-flex h-full w-full animate-ping rounded-full opacity-75 ${color}`} />
      <span className={`relative inline-flex h-2 w-2 rounded-full ${color}`} />
    </span>
  );
}

function StatusItem({
  status,
  label,
  isLoading,
  isError,
}: {
  status: number | undefined;
  label: string,
  isLoading: boolean;
  isError: boolean;
}) {
const { t } = useTranslation();
  const getStatus = (): StatusType => {
    if (isLoading) return 'pending';
    if (isError || status === undefined) return 'unknown';
    if (status >= 90) return 'error';
    if (status >= 75) return 'warning';
    return 'success';
  };

  return (
    <Tooltip>
      <TooltipTrigger onFocus={(e) => e.preventDefault()} className="cursor-default">
        <div className="flex items-center gap-2 px-2 py-1">
          <StatusIndicator status={getStatus()} />
          <span>{label}</span>
        </div>
      </TooltipTrigger>
      <TooltipContent side="left">
        {isLoading ? (
          <span>{t("Pending")}...</span>
        ) : status === undefined ? (
          <span>{t("Unknown")}</span>
        ) : (
          <span>{status}%</span>
        )}
      </TooltipContent>
    </Tooltip>
  );
}

function CPUStatusItem({ response }: { response: UseQueryResult<CPUStatus, Error> }) {
  const status = response.data?.cpu_usage;
  return (
    <StatusItem status={status} label="CPU" isLoading={response.isLoading} isError={response.isError} />
  )
}

function MemoryStatusItem({ response }: { response: UseQueryResult<MemoryStatus, Error> }) {
  const { t } = useTranslation();
  const status = response.data?.used_percent;
  return (
    <StatusItem status={status} label={t("Memory")} isLoading={response.isLoading} isError={response.isError} />
  )
}

function DiskStatusItem({ response }: { response: UseQueryResult<DiskStatus, Error> }) {
  const { t } = useTranslation();
  const status = response.data?.percent;
  return (
    <StatusItem status={status} label={t("Disk")} isLoading={response.isLoading} isError={response.isError} />
  )
}

export default function SystemStatus() {
  const { t } = useTranslation();
  const cpu = useGetSystemCPU();
  const memory = useGetSystemMemory();
  const disk = useGetSystemDisk();

  const queries = [cpu, memory, disk];
  const hasError = queries.some(q => q.isError);
  const isLoading = queries.some(q => q.isLoading);
  
  const getOverallStatus = (): StatusType => {
    if (hasError) return 'unknown';
    if (isLoading) return 'pending';

    const values = [
      cpu.data?.cpu_usage,
      memory.data?.used_percent,
      disk.data?.percent,
    ].filter((value) => value !== undefined);

    if (values.some(v => v >= 90)) return 'error';
    if (values.some(v => v >= 75)) return 'warning';

    return 'success';
  };

  const overallStatus = getOverallStatus();

  return (
    <Popover>
      <PopoverTrigger asChild>
        <Button variant="outline" size="sm" className="gap-2">
          <StatusIndicator status={overallStatus} />
          {t("System Status")}
        </Button>
      </PopoverTrigger>
      <PopoverContent className="grid grid-cols-1 p-1 w-[var(--radix-popover-trigger-width)]">
        <CPUStatusItem response={cpu} />
        <MemoryStatusItem response={memory} />
        <DiskStatusItem response={disk} />
      </PopoverContent>
    </Popover>
  );
}