import { useQuery } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import {
  UpdateCheck,
  SuggestReturn,
  CPUStatus,
  MemoryStatus,
  DiskStatus
} from "@/components/header/types";
import { useEffect, useState } from 'react';

export function useUpdateCheck() {
  return useQuery<UpdateCheck, AxiosError>({
    queryKey: ['updateCheck'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/system/update-check');
      return response.data;
    },
    staleTime: 24 * 60 * 60 * 1000, // 24 hours
  });
}

export function useSearchSuggest(value: string){
  const [debouncedInput, setDebouncedInput] = useState(value);
  const inputDelay = 100; // milliseconds

  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedInput(value);
    }, inputDelay);

    return () => {
      clearTimeout(timer);
    };
  }, [value, inputDelay]);

  return useQuery({
    queryKey: ['search', debouncedInput],
    queryFn: () => {
        return axios.get('/api/v1/search', {
          params: {
            term: debouncedInput,
          },
        });
    },
    enabled: value.length >= 3 && !!debouncedInput,
    select: (response) => {
        const data: Array<SuggestReturn> = response.data;
        return data;
    },
  });
}

export function useGetSystemCPU() {
  return useQuery<CPUStatus>({
    queryKey: ['system-status', 'system-cpu'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/system/status/cpu');
      return response.data;
    },
    refetchInterval: 2000,
  })
}

export function useGetSystemMemory() {
   return useQuery<MemoryStatus>({
    queryKey: ['system-status', 'system-memory'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/system/status/memory');
      return response.data;
    },
    refetchInterval: 8000,
  })
}

export function useGetSystemDisk() {
   return useQuery<DiskStatus>({
    queryKey: ['system-status', 'system-disk'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/system/status/root-drive');
      return response.data;
    },
    refetchInterval: 30000,
  })
}