import { Button } from "@/components/ui/button"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form'
import { useForm, SubmitHandler } from 'react-hook-form'
import React, { useState } from "react"
import { useTranslation } from "react-i18next"
import { useAddNdiff } from "@/components/nmap/ndiffs/queries"
import { ndiff } from "@/components/nmap/ndiffs/types"
import { useGetScans } from "@/components/nmap/scans/queries"
import { Combobox } from "@/components/ui/combobox"
import { useAuth } from "@/contexts/AuthContext";

export default function AddNdiffDialog({ children }: { children: React.ReactNode }) {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();
  const addNdiffMutation = useAddNdiff();
  const { data: scans = [] } = useGetScans();
  const { user } = useAuth();

  if (!user) return null;

  const form = useForm<ndiff>({
    defaultValues: {
      user_id: user.id,
      title: '',
      scan1_id: undefined,
      scan2_id: undefined,
    },
  });

  const onSubmit: SubmitHandler<ndiff> = (data) => {
    addNdiffMutation.mutate(data, {
      onSuccess: () => {
        form.reset();
        setOpen(false);
      },
    });
  };

  const handleDialogChange = (isOpen: boolean) => {
    setOpen(isOpen);
    if (!isOpen) {
      form.reset();
    }
  };
  
  return (
    <Dialog open={open} onOpenChange={handleDialogChange}>
      <DialogTrigger asChild>
        <div>{children}</div>
      </DialogTrigger>
      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle>{t('Add Ndiff')}</DialogTitle>
          <DialogDescription>{t('Compare two Nmap scans to see differences')}</DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
            <FormField
              control={form.control}
              name="title"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Title')}</FormLabel>
                  <FormControl>
                    <Input {...field} className="col-span-3 border-border" />
                  </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="scan1_id"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Base Scan')}</FormLabel>
                  <FormControl>
                    <Combobox
                      options={scans
                        .filter(scan => scan.status === 'Completed')
                        .map(scan => ({ value: scan.id.toString(), label: scan.title }))}
                      value={field.value ? field.value.toString() : ''}
                      onChange={val => field.onChange(val ? parseInt(val) : undefined)}
                      placeholder={t('Select base scan')}
                      className="col-span-3 w-full"
                    />
                  </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="scan2_id"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Comparison Scan')}</FormLabel>
                  <FormControl>
                    <Combobox
                      options={scans
                        .filter(scan => scan.status === 'Completed')
                        .map(scan => ({ value: scan.id.toString(), label: scan.title }))}
                      value={field.value ? field.value.toString() : ''}
                      onChange={val => field.onChange(val ? parseInt(val) : undefined)}
                      placeholder={t('Select comparison scan')}
                      className="col-span-3 w-full"
                    />
                  </FormControl>
                </FormItem>
              )}
            />
            <DialogFooter>
              <Button type="submit" className="justify-end">
                {t('Submit')}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  )
} 