import { Button } from "@/components/ui/button";
import { DataTable, DataTableRowAction } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { columns } from "@/components/nmap/ndiffs/columns";
import AddNdiffDialog from "@/components/nmap/ndiffs/AddNdiffDialog";
import DeleteNdiffDialog from "@/components/nmap/ndiffs/DeleteNdiffDialog";
import React from 'react';
import { useNmapIntegrationUpdates } from "@/components/nmap/NmapIntegrationUpdatesProvider";
import { ndiff } from "@/components/nmap/ndiffs/types";
import { Plus } from "lucide-react";
import { useTranslation } from 'react-i18next';
import { useAuth } from "@/contexts/AuthContext";
import { hasPermission } from "@/components/role/utils";

type NdiffsProps = {
  data: ndiff[] | undefined;
  isLoading: boolean;
}

export default function Ndiffs({ data, isLoading }: NdiffsProps) {
  const { t } = useTranslation();
  const { mergeNdiff } = useNmapIntegrationUpdates();
  const mergedData = (data || []).map(mergeNdiff);
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [idsToDelete, setIdsToDelete] = React.useState<number[]>([]);
  const { user } = useAuth();

  const actions: DataTableRowAction<ndiff>[] = [];

  if (hasPermission(user?.role, 'nmap_permissions', 'ndiffs', 'delete')) {
    actions.push(
      {
        label: t("Delete"),
        value: "delete",
        onClick: async (selectedRows: any) => {
          setIdsToDelete(selectedRows.map((row: any) => row.id));
          setDeleteDialogOpen(true);
        }
      }
    );
  }

  return (
    <>
      {hasPermission(user?.role, 'nmap_permissions', 'ndiffs', 'post') &&
        <div className="pb-1 flex">
          <AddNdiffDialog>
            <Button>
              <Plus />
              {t('Create Ndiff')}
            </Button>
          </AddNdiffDialog>
        </div>
      }
      {!isLoading && (
        <DataTable 
          columns={columns} 
          data={mergedData} 
          sorting={[{ id: "Created At", desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            'Duration': false,
            'Finished At': false,
          }}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <DeleteNdiffDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteNdiffDialog>
    </>
  )
}