import { ColumnDef } from "@tanstack/react-table"
import { ndiff } from "@/components/nmap/ndiffs/types"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import { useNavigate } from "react-router-dom"
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import type { VariantProps } from "class-variance-authority"
import { badgeVariants } from "@/components/ui/badge"
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip"
import { Button } from "@/components/ui/button"
import { 
  Eye,
  MoreHorizontal,
  Trash2,
  Download,
  RefreshCw,
} from "lucide-react"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { useCallback, useState } from "react"
import DeleteNdiffDialog from "@/components/nmap/ndiffs/DeleteNdiffDialog"
import { 
  useDownloadNdiff,
  useAddNdiff,
} from "@/components/nmap/ndiffs/queries"
import { useAuth } from "@/contexts/AuthContext"
import { Duration } from "@/components/ui/Duration";
import { useTranslation } from 'react-i18next';
import { hasPermission } from "@/components/role/utils";

type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

export const columns: ColumnDef<ndiff>[] = [
  {
    id: "select",
    header: ({ table }) => (
      <Checkbox
        checked={
          table.getIsAllPageRowsSelected() ||
          (table.getIsSomePageRowsSelected() && "indeterminate")
        }
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
  },
  {
    accessorKey: "title",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Ndiff Name" />
    },
  },
  {
    id: "view",
    cell: ({ row }) => {
      const scan = row.original
      const navigate = useNavigate();
      const { t } = useTranslation();
  
      function viewScan(id: number) {
        navigate(`/nmap/ndiff/${id}`);
      }
  
      return (
        scan.status === "Completed" && (
          <TooltipProvider>
          <Tooltip>
            <TooltipTrigger asChild>
              <Button
              variant="ghost"
              className="h-8 w-8 p-0"
              onClick={() => {
                viewScan(scan.id);
              }}
              >
                <Eye />
                <span className="sr-only">{t('View Ndiff results')}</span>
              </Button>
            </TooltipTrigger>
            <TooltipContent side="right">
              <p>{t('View Ndiff Results')}</p>
            </TooltipContent>
          </Tooltip>
          </TooltipProvider>
        )
      )
    },
  },
  {
    accessorKey: "status",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Status" />
    },
    cell: ({ row }) => {
      const status = row.getValue("status")
      let variant: BadgeVariant = "default"
  
      if (status === "Completed") {
        variant = "green"
      } else if (status === "In Progress" || status === "Pending") {
        variant = "blue"
      } else if (status === "Failed") {
        variant = "red"
      }
      return <Badge variant={variant}>{String(status)}</Badge>
    },
  },
  {
    id: "Scan 1",
    accessorFn: row => row.scan1?.title ?? "N/A",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scan 1" />
    },
    cell: ({ row }) => {
      const scan1 = row.original.scan1;
      const navigate = useNavigate();

      return (
        <button
          onClick={() => scan1 && navigate(`/nmap/scan/${scan1.id}`)}
          className="hover:underline"
        >
          {scan1?.title ?? "N/A"}
        </button>
      );
    },
  },
  {
    id: "Scan 2",
    accessorFn: row => row.scan2?.title ?? "N/A",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scan 2" />
    },
    cell: ({ row }) => {
      const scan2 = row.original.scan2;
      const navigate = useNavigate();

      return (
        <button
          onClick={() => scan2 && navigate(`/nmap/scan/${scan2.id}`)}
          className="hover:underline"
        >
          {scan2?.title ?? "N/A"}
        </button>
      );
    },
  },
  {
    id: "Duration",
    accessorKey: "duration",
    header: ({ column }) => <DataTableColumnHeader column={column} title="Duration" />,
    cell: ({ row }) => {
      const ndiff = row.original;
      return (
        <Duration
          started_at={ndiff.started_at as string | undefined}
          finished_at={ndiff.finished_at as string | undefined}
        />
      );
    },
  },
  {
    id: "Created At",
    accessorKey: "created_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Created At" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const createdAtValue = row.getValue("Created At");
      
      if (!createdAtValue || (typeof createdAtValue !== "string")) {
        return <div>{t('Ndiff Pending')}...</div>;
      }

      const createdAt = new Date(createdAtValue);
      return (
        <div>
          {createdAt.toLocaleString()}
        </div>
      );
    },
  },
  {
    id: "Finished At",
    accessorKey: "finished_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Finished At" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const finishedAtValue = row.getValue("Finished At");

      if (!finishedAtValue || typeof finishedAtValue !== "string") {
        return <div>{t('Not Finished')}</div>;
      }

      const finishedAt = new Date(finishedAtValue);
      return (
        <div>
          {finishedAt.toLocaleString()}
        </div>
      );
    },
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const ndiff = row.original
      const { mutate: downloadNdiff } = useDownloadNdiff()
      const [deleteDialogOpen, setDeleteDialogOpen] = useState(false)
      const addNdiffMutation = useAddNdiff();
      const { user } = useAuth();
      const { t } = useTranslation();
  
      const handleDownload = useCallback((ext: "xml" | "json" | "txt") => {
        const filename = `${ndiff.output_filename}.${ext}`
        downloadNdiff(filename, {
          onSuccess: (data: Blob) => {
            const url = window.URL.createObjectURL(new Blob([data], { type: ext === "xml" ? "application/xml" : "text/plain" }))
            const link = document.createElement("a")
            link.href = url
            link.setAttribute("download", `${ndiff.output_filename}.${ext}`)
            document.body.appendChild(link)
            link.click()
            link.parentNode?.removeChild(link)
            window.URL.revokeObjectURL(url)
          },
        })
      }, [downloadNdiff, ndiff])

      const handleRerunNdiff = () => {
        if (!user) return;
        addNdiffMutation.mutate({
          user_id: user.id,
          title: ndiff.title,
          scan1_id: ndiff.scan1_id,
          scan2_id: ndiff.scan2_id,
        });
      };
  
      return (
        <>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" className="h-8 w-8 p-0">
                <span className="sr-only">{t('Open menu')}</span>
                <MoreHorizontal className="h-4 w-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="center">
              {hasPermission(user?.role, 'nmap_permissions', 'ndiffs', 'post') &&
                <>
                  <DropdownMenuItem onClick={handleRerunNdiff}>
                    <RefreshCw />
                    {t('Rerun Ndiff')}
                  </DropdownMenuItem>
                  <DropdownMenuSeparator />
                </>
              }
              <DropdownMenuItem onClick={() => handleDownload("xml")}>
                <Download />
                {t('Download XML')}
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => handleDownload("json")}>
                <Download />
                {t('Download JSON')}
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => handleDownload("txt")}>
                <Download />
                {t('Download Text')}
              </DropdownMenuItem>
              {hasPermission(user?.role, 'nmap_permissions', 'ndiffs', 'delete') &&
                <>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem onClick={() => setDeleteDialogOpen(true)}>
                    <Trash2 />
                    {t('Delete')}
                  </DropdownMenuItem>
                </>
              }
            </DropdownMenuContent>
          </DropdownMenu>
          <DeleteNdiffDialog ids={[ndiff.id]} open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
            <span />
          </DeleteNdiffDialog>
        </>
      )
    },
  },
]