import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";
import { ndiff } from '@/components/nmap/ndiffs/types';

export function useGetNdiffs() {
  return useQuery<any[], AxiosError>({
    queryKey: ['ndiffs'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/nmap/ndiffs');
      return response.data;
    },
    staleTime: 5 * 60 * 1000, // 5 minutes
  })
}

export function useGetNdiff(id: string) {
  return useQuery({
    queryKey: ['ndiff', id],
    queryFn: async () => {
      const response = await axios.get(`/api/v1/nmap/ndiffs/${id}`)
      return response.data;
    },
    enabled: !!id,
  })
}

export function useAddNdiff() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Omit<ndiff, 'id' | 'created_at' | 'updated_at' | 'finished_at' | 'status' | 'output_filename' | 'ndiff_results'>) => {
      return axios.post(
        '/api/v1/nmap/ndiffs',
        {
          user_id: data.user_id,
          title: data.title,
          scan1_id: data.scan1_id,
          scan2_id: data.scan2_id,
        },
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['ndiffs'] });
      toast(t('Ndiff submitted.'),
        { description: "" });
    },
    onError: () => {
      toast(t('Failed to submit Ndiff.'),
        { description: t('Please try again.') });
    },
  });
}

export function useDeleteNdiffs() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (ndiffIds: number[]) => {
      return axios.delete(
        '/api/v1/nmap/ndiffs',
        {
          data: { ids: ndiffIds },
        }
      );
    },
    onMutate: async (ndiffIds) => {
      const key = ['ndiffs'];
      await queryClient.cancelQueries({ queryKey: key });
      const previousNdiffs = queryClient.getQueryData<ndiff[]>(key);
      queryClient.setQueryData(key, (old: ndiff[] = []) =>
        old.filter((ndiff) => !ndiffIds.includes(ndiff.id))
      );
      return { previousNdiffs };
    },
    onSuccess: (_data, ndiffIds) => {
      const message = ndiffIds.length > 1
        ? t('Ndiffs deleted.')
        : t('Ndiff deleted.');
      toast(message);
    },
    onError: (_error, ndiffIds, context) => {
      queryClient.setQueryData(['ndiffs'], context?.previousNdiffs)
      const errorMessage = ndiffIds.length > 1
        ? t('Failed to delete Ndiffs.')
        : t('Failed to delete Ndiff.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['ndiffs'] });
    },
  });
}

export function useDownloadNdiff() {
  return useMutation({
    mutationFn: async (filename: string) => {
      const response = await axios.get(
        `/api/v1/nmap/ndiffs/${filename}/download`,
        { responseType: 'blob' }
      );
      return response.data;
    },
  });
}
