import { Bar, BarChart, CartesianGrid, XAxis, Cell } from "recharts";
import React from "react";
import { useTranslation } from "react-i18next";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart";
import { DailyScanCount } from "@/components/nmap/overview/types";

const baseChartConfig = {
  total_scans: {
    label: "Total Scans",
  },
} satisfies ChartConfig;

interface Props {
  data: DailyScanCount[];
}

export function NmapScansOverTimeBarChart({ data }: Props) {
  const { t } = useTranslation();
  const chartData: (DailyScanCount & { fill: string })[] = React.useMemo(() => {
    if (!data) return [];
    return data.map((item, index) => ({
      ...item,
      fill: `var(--chart-${(index % 10) + 1})`,
    }));
  }, [data]);

  const dynamicChartConfig: ChartConfig = React.useMemo(() => {
    return {
      ...baseChartConfig,
      ...chartData.reduce<Record<string, { label: string; color: string }>>(
        (acc, item) => {
          acc[item.date] = { label: item.date, color: item.fill };
          return acc;
        },
        {}
      ),
    };
  }, [chartData]);

  return (
    <Card className="flex flex-col">
      <CardHeader className="items-center pb-0">
        <CardTitle>{t("Scans Over Last 7 Days")}</CardTitle>
        <CardDescription>
          {t("Completed scans over the past week")}
        </CardDescription>
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        <ChartContainer
          config={dynamicChartConfig}
          className="mx-auto aspect-auto h-[200px] w-full"
        >
          <BarChart
            data={chartData}
            margin={{ top: 10, right: 10, left: 10, bottom: 10 }}
          >
            <CartesianGrid vertical={false} strokeDasharray="3 3" />
            <XAxis
              dataKey="date"
              tickLine={false}
              axisLine={false}
              tickMargin={8}
              angle={-45}
              textAnchor="end"
              height={60}
            />
            <ChartTooltip
              content={
                <ChartTooltipContent
                  className="w-[150px]"
                  nameKey="total_scans"
                />
              }
            />
            <Bar dataKey="total_scans" radius={[4, 4, 0, 0]}>
              {chartData.map((entry) => (
                <Cell key={entry.date} fill={entry.fill} />
              ))}
            </Bar>
          </BarChart>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col gap-2 text-sm" />
    </Card>
  );
}
