import { Bar, BarChart, CartesianGrid, XAxis, YAxis } from "recharts"
import React from "react"
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import {
  ChartConfig,
  ChartContainer,
  ChartLegend,
  ChartLegendContent,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart"
import { useTranslation } from "react-i18next"

interface ScheduledScan {
  name: string
  days: string[]
  runs: number
}

interface Props {
  data: ScheduledScan[]
}


export function ScheduledScansChart({ data }: Props) {
  const { t } = useTranslation();

  const DAYS = [
    t("Sunday"),
    t("Monday"),
    t("Tuesday"),
    t("Wednesday"),
    t("Thursday"),
    t("Friday"),
    t("Saturday"),
  ]
  
  const chartData = React.useMemo(() => {
    return DAYS.map((day, index) => {
      const row: Record<string, any> = { day }
      data.forEach((scan) => {
        row[scan.name] = scan.days.includes(index.toString())
          ? scan.runs
          : 0
      })
      return row
    })
  }, [data])

  const chartConfig: ChartConfig = React.useMemo(() => {
    return data.reduce<Record<string, { label: string; color: string }>>(
      (acc, scan, idx) => {
        acc[scan.name] = {
          label: scan.name,
          color: `var(--chart-${(idx % 10) + 1})`,
        }
        return acc
      },
      {}
    )
  }, [data])

  return (
    <Card className="flex flex-col">
      <CardHeader className="items-center pb-0">
        <CardTitle>{t("Scheduled Scans")}</CardTitle>
        <CardDescription>
          {t("Number of scans that are scheduled to run per day")}
        </CardDescription>
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        <ChartContainer
          config={chartConfig}
          className="mx-auto aspect-auto h-[225px] w-full"
        >
          <BarChart
            data={chartData}
            margin={{ top: 10, right: 10, left: 10, bottom: 10 }}
          >
            <CartesianGrid vertical={false} strokeDasharray="3 3" />
            <XAxis
              dataKey="day"
              tickLine={false}
              axisLine={false}
              tickMargin={8}
              angle={-45}
              textAnchor="end"
              height={60}
              interval={0}
            />
            <YAxis
              label={{
                value: t("Scans per Day"),
                angle: -90,
                position: "insideLeft",
                offset: -5,
                style: { textAnchor: "middle" }
              }}
              tickLine={false}
              axisLine={false}
            />
            <ChartTooltip
              content={
                <ChartTooltipContent
                  hideLabel
                  className="w-[180px]"
                  formatter={(value, name) => {
                    const color =
                      chartConfig[name as keyof typeof chartConfig]?.color || "#000"
                    return (
                      <>
                        <div
                          className="h-2.5 w-2.5 shrink-0 rounded-[2px]"
                          style={{ backgroundColor: color }}
                        />
                        {chartConfig[name as keyof typeof chartConfig]?.label || name}
                        <div className="text-foreground ml-auto flex items-baseline gap-0.5 font-mono font-medium tabular-nums">
                          {value}
                          <span className="text-muted-foreground font-normal">{t('scans')}</span>
                        </div>
                      </>
                    )
                  }}
                />
              }
            />
            <ChartLegend content={<ChartLegendContent />} />
            {data.map((scan, idx) => (
              <Bar
                key={scan.name}
                dataKey={scan.name}
                stackId="a"
                fill={`var(--chart-${(idx % 10) + 1})`}
                radius={[4, 4, 0, 0]}
              />
            ))}
          </BarChart>
        </ChartContainer>
      </CardContent>
    </Card>
  )
}
