import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form'
import { useForm, SubmitHandler } from 'react-hook-form'
import React, { useState, useRef } from "react"
import { useTranslation } from "react-i18next"
import { useAddProfile } from "@/components/nmap/profiles/queries";
import { nmapOptionField, ProfileFormValues, ProfilePayload } from "@/components/nmap/profiles/types";
import { validateScanHasInterfaceFlag, validateScanParameters } from '@/components/nmap/utils';
import { Checkbox } from "@/components/ui/checkbox";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetFooter,
  SheetTitle,
  SheetDescription,
  SheetTrigger,
} from "@/components/ui/sheet";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible"
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import {  
  ChevronDown,
  ChevronRight,
  Info,
} from "lucide-react";
import { Select, SelectTrigger, SelectContent, SelectGroup, SelectItem, SelectValue } from "@/components/ui/select";
import { PageSeparator } from "@/components/Page"
import { NMAP_OPTION_SECTIONS } from "@/components/nmap/profiles/nmapOptionSections"

export default function AddProfileSheet({ children }: { children: React.ReactNode }) {
  const { t } = useTranslation();
  const [open, setOpen] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [openSections, setOpenSections] = useState<string[]>([]);
  const [selectedOptions, setSelectedOptions] = useState<{ [key: string]: string }>({});
  const [parametersString, setParametersString] = useState("");
  const debounceTimeout = useRef<NodeJS.Timeout | null>(null);
  const inputDebounceTimeout = useRef<NodeJS.Timeout | null>(null);
  const initializedRef = useRef(false);
  const [inputDrafts, setInputDrafts] = useState<{ [key: string]: string }>({});
  const addProfileMutation = useAddProfile();

  const form = useForm<ProfileFormValues>({
    defaultValues: {
        name: '',
        parameters: '',
        description: '',
        tags: '',
    },
  }); 

  const NMAP_OPTION_MAP = NMAP_OPTION_SECTIONS.flatMap(section => section.fields);
  const NMAP_OPTION_FIELD_MAP: { [key: string]: nmapOptionField } = {};
  for (const field of NMAP_OPTION_MAP) {
    if (field.key) {
      NMAP_OPTION_FIELD_MAP[field.key] = field;
    }
    if (field.type === 'select' && field.options) {
      for (const option of field.options) {
        NMAP_OPTION_FIELD_MAP[option.key] = field;
      }
    }
  }

  const getFieldForKey = (key: string) => {
    return NMAP_OPTION_FIELD_MAP[key];
  };

  const shouldIncludeOption = (key: string, value: string) => {
    const field = getFieldForKey(key);
    if (!field) return false;
    if (field.type === 'checkbox') {
      return value !== undefined;
    }
    if (field.type === 'input') {
      return value !== undefined && value !== null && value !== "";
    }
    if (field.type === 'select') {
      if (field.key) {
        return value !== undefined && value !== null && value !== "";
      } else {
        return true;
      }
    }
    return true;
  };
  
  const generateParametersString = (selectedOpts: { [key: string]: string }) => {
    const formattedOptions = Object.entries(selectedOpts)
      .filter(([key, value]) => shouldIncludeOption(key, value))
      .map(([key, value]) => {
        if (key === 'targets') return value;
        if (['-PS', '-PA', '-PU', '-PY', '-PO', '-T'].includes(key)) {
          return `${key}${value}`;
        }
        return `${key} ${value}`.trim();
      });

    const result = formattedOptions.join(" ");
    setParametersString(result);
    form.setValue("parameters", parametersString);
    return result;
  };

  // Parse function reused by change handler and initial population
const parseParameters = (value: string) => {
  const tokens =
    value.match(/--?\w+(?:-\w+)*|-[A-Za-z]+[^\s]*|[^\s]+/g) || [];
  const allOptionKeys = [
    ...NMAP_OPTION_MAP.filter(o => o.key).map(o => o.key!),
    ...NMAP_OPTION_MAP
      .filter(o => o.type === "select" && o.options)
      .flatMap(o => o.options!.map(opt => opt.key)),
  ];

  const newSelected: { [key: string]: string } = {};
  const usedIndices = new Set<number>(); // Track tokens already used as option arguments
  let i = 0;
  let targets: string[] = [];

  while (i < tokens.length) {
    const token = tokens[i];

    // Handle long options (--something)
    if (token.startsWith("--")) {
      const opt = NMAP_OPTION_MAP.find(
        (o: nmapOptionField) => o.key === token
      );

      if (opt && (opt.type === "input" || opt.type === "select")) {
        const next = tokens[i + 1];
        if (next && !allOptionKeys.includes(next)) {
          newSelected[opt.key!] = next;
          usedIndices.add(i + 1); // mark argument as used
          i += 2;
          continue;
        } else {
          newSelected[opt.key!] = "";
          i += 1;
          continue;
        }
      } else if (opt) {
        newSelected[opt.key!] = "";
        i += 1;
        continue;
      }
    }
    
    // Handle short options (-e, -p80, etc.)
    if (token.startsWith("-") && !token.startsWith("--")) {
      const opt = NMAP_OPTION_MAP.find(
        (o: nmapOptionField) => o.key && token.startsWith(o.key)
      );
      if (opt && (opt.type === "input" || opt.type === "select")) {
        const next = tokens[i + 1];
        if (next && !allOptionKeys.includes(next)) {
          newSelected[opt.key!] = next;
          usedIndices.add(i + 1); // mark argument as used
          i += 2;
          continue;
        } else {
          newSelected[opt.key!] = "";
          i += 1;
          continue;
        }
      } else if (opt) {
        newSelected[opt.key!] = "";
        i += 1;
        continue;
      }

      // Handle grouped or sub-select flags (e.g., -sS, -sU, etc.)
      let matchedFlagGroup = false;
      for (const field of NMAP_OPTION_MAP.filter(
        f => f.type === "select" && !f.key && f.options
      )) {
        const match = field.options!.find(o => o.key === token);
        if (match) {
          newSelected[match.key] = "";
          matchedFlagGroup = true;
          i += 1;
          break;
        }
      }
      if (matchedFlagGroup) continue;
    }

    // Handle positional targets (IP addresses, hostnames, etc.)
    if (!allOptionKeys.includes(token) && !usedIndices.has(i)) {
      targets.push(token);
    }
    i += 1;
    }
    if (targets.length > 0) {
      newSelected["targets"] = targets.join(" ");
    }
    return newSelected;
  };

  const handleDialogChange = (isOpen: boolean) => {
    setOpen(isOpen);
    if (!isOpen) {
      initializedRef.current = false;
       form.reset();
      setSelectedOptions({});
      setParametersString("");
      setError(null);
      setOpenSections([]);
    }
  };
 
  const handleParametersChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setParametersString(value);
    if (debounceTimeout.current) clearTimeout(debounceTimeout.current);
    
    debounceTimeout.current = setTimeout(() => {
      const newSelected = parseParameters(value);
      setSelectedOptions(newSelected);
      form.setValue("parameters", value);
    }, 750);
  };

  const onSubmit: SubmitHandler<ProfileFormValues> = (data) => {
    if (!validateScanParameters(data.parameters || "", setError)) return;
    if (!validateScanHasInterfaceFlag(data.parameters || "", setError, true)) return;
    const tags = (typeof data.tags === 'string' ? data.tags : '').split(',').map((t: string) => t.trim()).filter(Boolean);

    const payload: ProfilePayload = {
      name: data.name,
      parameters: data.parameters || "",
      description: data.description || "",
      tags,
    };
    // create new
    addProfileMutation.mutate(payload, {
      onSuccess: () => {
        form.reset();
        setOpen(false);
        setSelectedOptions({});
        setParametersString("");
        setError(null);
        setOpenSections([]);
      },
    });
    
  };

  return (
    <Sheet open={open} onOpenChange={handleDialogChange}>
      <SheetTrigger asChild>
        <div>{children}</div>
      </SheetTrigger>
      <Form {...form}>
        <form id="form" onSubmit={form.handleSubmit(onSubmit)} className="">
          <SheetContent className="sm:max-w-[800px] flex flex-col h-full">
            <SheetHeader>
              <SheetTitle>{t('Add Profile')}</SheetTitle>
              <SheetDescription>{t('Use this to create a new Nmap Profile')}</SheetDescription>
              {error && <div className="text-destructive text-sm mb-2">{error}</div>}
              <div className="my-2 p-2 bg-muted rounded border text-xs">
                <div className="font-semibold mb-1">{t('Scan Parameters')}</div>
                <FormField
                  control={form.control}
                  name="parameters"
                  render={({ field }) => (
                    <FormItem>
                      <FormControl>
                        <Input
                          {...field}
                          value={parametersString}
                          onChange={(e) => {
                            handleParametersChange(e);
                            field.onChange(e.target.value);
                          }}
                          placeholder={t('Nmap parameters will appear here') as string}
                          className="font-mono text-sm bg-background border-border"
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </div>
            </SheetHeader>
            <ScrollArea className="min-h-0 flex flex-col pr-4">
              <div className="grid gap-4 mr-4">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0 pt-2">
                      <FormLabel className="text-right">
                        {t('Profile Name')}
                      </FormLabel>
                      <FormControl>
                        <Input {...field} className="col-span-3 border-border" />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                      <FormLabel className="text-right">
                        {t('Description')}
                      </FormLabel>
                      <FormControl>
                        <Input {...field} className="col-span-3 border-border" />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="tags"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                      <FormLabel className="text-right">
                        {t('Tags')}
                      </FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          className="col-span-3 border-border"
                          placeholder={t('Comma separated') as string}
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </div>
              <PageSeparator />
              <div className="mb-4">
                <div className="font-semibold mb-2">{t('Nmap Options')}</div>
                <div className="space-y-2">
                  {NMAP_OPTION_SECTIONS.map(sectionObj => (
                    <Collapsible
                      key={sectionObj.section}
                      open={openSections.includes(sectionObj.section)}
                      onOpenChange={open => {
                        setOpenSections(prev => {
                          if (open) {
                            return prev.includes(sectionObj.section) ? prev : [...prev, sectionObj.section];
                          } else {
                            return prev.filter(section => section !== sectionObj.section);
                          }
                        });
                      }}
                    >
                      <CollapsibleTrigger className="flex items-center justify-between w-full px-3 py-2 text-left border rounded bg-muted hover:bg-muted/80">
                        <span className="font-semibold">{sectionObj.section}</span>
                        <span className="text-sm flex items-center ml-2">
                          {openSections.includes(sectionObj.section) ? (
                            <ChevronDown size={18} />
                          ) : (
                            <ChevronRight size={18} />
                          )}
                        </span>
                      </CollapsibleTrigger>
                      <CollapsibleContent className="px-2 pb-2 pt-2">
                        <div className="grid gap-2">
                          {sectionObj.fields.map((opt: nmapOptionField) => (
                            <div key={opt.label} className="grid grid-cols-4 items-center">
                              <div className="col-span-2 flex items-center">
                                <label htmlFor={opt.key ? `nmap-opt-${opt.key}` : undefined} className="text-sm flex items-center gap-2">
                                  <span>
                                    {t(opt.label)}
                                    {opt.key && (
                                      <span className="text-muted-foreground font-mono text-xs ml-1">
                                        ({opt.key})
                                      </span>
                                    )}
                                  </span>
                                  {opt.tooltip && (
                                    <Tooltip>
                                      <TooltipTrigger asChild>
                                        <span tabIndex={0}><Info size={16} /></span>
                                      </TooltipTrigger>
                                      <TooltipContent className="max-w-80 text-wrap">
                                        {t(opt.tooltip)}
                                      </TooltipContent>
                                    </Tooltip>
                                  )}
                                </label>
                              </div>
                              <div className="col-span-2 flex items-center">
                                {opt.type === "checkbox" && opt.key && (
                                  <Checkbox
                                    checked={selectedOptions[opt.key] !== undefined}
                                    onCheckedChange={checked => {
                                      setSelectedOptions(prev => {
                                        const newOpts = { ...prev };
                                        if (checked) {
                                          newOpts[opt.key!] = prev[opt.key!] || "";
                                        } else {
                                          delete newOpts[opt.key!];
                                        }
                                        generateParametersString(newOpts);
                                        return newOpts;
                                      });
                                    }}
                                    id={`nmap-opt-${opt.key}`}
                                  />
                                )}
                                {opt.type === "input" && opt.key && (
                                  <Input
                                    className="w-15/16"
                                    placeholder={t(opt.placeholder ?? '')}
                                    value={
                                      inputDrafts[opt.key!] !== undefined
                                        ? inputDrafts[opt.key!]
                                        : selectedOptions[opt.key!] || ""
                                    }
                                    onChange={e => {
                                      const val = e.target.value;
                                      setInputDrafts(drafts => ({ ...drafts, [opt.key!]: val }));

                                      // Debounced update while typing
                                      if (inputDebounceTimeout.current) clearTimeout(inputDebounceTimeout.current);
                                      inputDebounceTimeout.current = setTimeout(() => {
                                        setSelectedOptions(prev => ({ ...prev, [opt.key!]: val }));
                                        generateParametersString({ ...selectedOptions, [opt.key!]: val });
                                      }, 750);
                                    }}
                                    onBlur={() => {
                                      setTimeout(() => { // Delay state updates to prevent interfering with browser tabbing
                                        const val =
                                          inputDrafts[opt.key!] !== undefined
                                            ? inputDrafts[opt.key!]
                                            : selectedOptions[opt.key!] || "";
                                        setSelectedOptions(prev => ({ ...prev, [opt.key!]: val }));
                                        generateParametersString({ ...selectedOptions, [opt.key!]: val });
                                        setInputDrafts(drafts => {
                                          const newDrafts = { ...drafts };
                                          delete newDrafts[opt.key!];
                                          return newDrafts;
                                        });
                                      }, 0);
                                    }}
                                  />
                                )}
                                {opt.type === "select" && opt.options && (
                                  <Select
                                    value={(() => {
                                      if (!opt.options) return "";
                                      // For selects with a key, value is stored under key
                                      if (opt.key) return selectedOptions[opt.key] || "";
                                      // For flag-group selects, value is the key of the selected option
                                      const found = opt.options.find(o => selectedOptions[o.key] !== undefined);
                                      return found ? found.key : "";
                                    })()}
                                    onValueChange={val => {
                                      setSelectedOptions(prev => {
                                        const newOpts = { ...prev };
                                        if (opt.options) {
                                          if (opt.key) {
                                            newOpts[opt.key] = val || "";
                                          } else {
                                            opt.options.forEach(o => { delete newOpts[o.key]; });
                                            if (val) {
                                              newOpts[val] = "";
                                            }
                                          }
                                        }
                                        generateParametersString(newOpts);
                                        return newOpts;
                                      });
                                    }}
                                  >
                                    <SelectTrigger className="w-15/16 max-h-60">
                                      <SelectValue placeholder={t(opt.label)} />
                                    </SelectTrigger>
                                    <SelectContent className="max-h-60">
                                      <SelectGroup>
                                        {opt.options.map(option => (
                                          <Tooltip key={option.key}>
                                            <TooltipTrigger asChild>
                                              <SelectItem value={option.key}>{option.label}</SelectItem>
                                            </TooltipTrigger>
                                            {option.tooltip && (
                                              <TooltipContent className="max-w-60 text-wrap" side="left">
                                                {option.tooltip}
                                              </TooltipContent>
                                            )}
                                          </Tooltip>
                                        ))}
                                      </SelectGroup>
                                    </SelectContent>
                                  </Select>
                                )}
                              </div>
                            </div>
                          ))}
                        </div>
                      </CollapsibleContent>
                    </Collapsible>
                  ))}
                </div>
              </div>
            </ScrollArea>
            <SheetFooter>
              <Button type="submit" form="form" className="justify-end">
                  {t('Submit')}
              </Button>
            </SheetFooter>
          </SheetContent>
        </form>
      </Form>
    </Sheet>
  )
}