import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form';
import { useForm, SubmitHandler } from 'react-hook-form';
import { useState } from "react";
import { useTranslation } from "react-i18next";
import { useAddScan } from "@/components/nmap/scans/queries";
import { nmapProfile } from "@/components/nmap/profiles/types";
import { validateScanHasInterfaceFlag, validateScanParameters } from '@/components/nmap/utils';

type RunScanDialogProps = {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  profile: nmapProfile;
}

type FormValues = {
  target: string;
};

export default function RunScanDialog({ open, onOpenChange, profile }: RunScanDialogProps) {
  const { t } = useTranslation();
  const addScanMutation = useAddScan();
  const [error, setError] = useState<string | null>(null);

  const form = useForm<FormValues>({
    defaultValues: {
      target: '',
    },
  });

  const onSubmit: SubmitHandler<FormValues> = (data) => {
    const scan_parameters = `${profile.parameters} ${data.target}`.trim();
    if (!validateScanParameters(scan_parameters, setError)) return;
    if (!validateScanHasInterfaceFlag(scan_parameters || "", setError)) return;
    addScanMutation.mutate({
      title: `${profile.name}`,
      scan_parameters,
    }, {
      onSuccess: () => {
        form.reset();
        onOpenChange(false);
      },
    });
  };

  const handleDialogChange = (isOpen: boolean) => {
    onOpenChange(isOpen);
    if (!isOpen) {
      form.reset();
    }
  };

  return (
    <Dialog open={open} onOpenChange={handleDialogChange}>
      <DialogContent className="sm:max-w-[425px]">
        <DialogHeader>
          <DialogTitle>{t('Run Scan With Profile')}</DialogTitle>
          <DialogDescription>{t('Specify a target (IP address or range) to run a scan with this profile.')}</DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
            {error && <div className="text-destructive text-sm mb-2">{error}</div>}
            <FormField
              control={form.control}
              name="target"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Target')}</FormLabel>
                  <FormControl>
                    <Input {...field} className="col-span-3 border-border" placeholder={t('e.g., 192.168.1.0/24 or 10.0.0.1')} />
                  </FormControl>
                </FormItem>
              )}
            />
            <DialogFooter>
              <Button type="submit" className="justify-end" disabled={addScanMutation.isPending}>
                {addScanMutation.isPending ? t('Submitting...') : t('Run Scan')}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
} 