import { ColumnDef } from "@tanstack/react-table"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import { Button } from "@/components/ui/button"
import { 
  MoreHorizontal,
  Trash2,
  Pencil,
  CopyPlus,
  Plus,
  CalendarPlus,
} from "lucide-react"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { HoverCard, HoverCardContent, HoverCardTrigger } from "@/components/ui/hover-card"
import { nmapProfile } from "@/components/nmap/profiles/types"
import { useAddProfile } from "@/components/nmap/profiles/queries"
import { useTranslation } from "react-i18next"
import { useAuth } from "@/contexts/AuthContext"
import { hasPermission } from "@/components/role/utils"

export const columns = (
  data: nmapProfile[],
  handleRunScan: (nmapProfile: nmapProfile) => void,
  handleAddScheduledScan: (nmapProfile: nmapProfile) => void,
  handleEditProfile: (nmapProfile: nmapProfile) => void,
  handleDelete: (id: number) => void,
) : ColumnDef<nmapProfile>[] => [
  {
    id: "select",
    header: ({ table }) => (
      <Checkbox
        checked={
          table.getIsAllPageRowsSelected() ||
          (table.getIsSomePageRowsSelected() && "indeterminate")
        }
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
  },
  {
    accessorKey: "name",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Profile Name" />
    },
    cell: ({ row }) => <span>{row.getValue("name")}</span>,
  },
  {
    id: "parameters",
    accessorKey: "parameters",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Parameters" />
    },
    cell: ({ row }) => {
      const parameters = row.getValue("parameters") as string;
      return (
        <div className="max-w-[400px] break-words whitespace-pre-wrap">
          {parameters}
        </div>
      );
    },
  },
  {
    id: "description",
    accessorKey: "description",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Description" />
    },
    cell: ({ row }) => {
      const description = row.getValue("description") as string | undefined;
      if (!description) return <span>-</span>;
      return (
        <div className="max-w-[700px] break-words whitespace-pre-wrap">
          {description}
        </div>
      );
    },
  },
  {
    id: "tags",
    accessorKey: "tags",
    accessorFn: (row) => Array.isArray(row.tags) ? row.tags.join(" ") : "",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Tags" />
    },
    cell: ({ row }) => {
      const tags = row.original.tags as string[] | undefined;
      if (!tags || tags.length === 0) return <span>-</span>;
      const sortedTags = [...tags].sort((a, b) => a.localeCompare(b));
      return (
        <div className="flex items-center gap-1">
          {sortedTags.map((tag, index, arr) => {
            if (index < 4) return <Badge key={index} variant="outline">{tag}</Badge>;
            else if (index === 4) return (
              <HoverCard openDelay={50} closeDelay={50} key={index}>
                <HoverCardTrigger>
                  <Badge variant="outline" className="hover:cursor-default">+ {arr.length - 4}</Badge>
                </HoverCardTrigger>
                <HoverCardContent side="right" className="p-1.5 w-fit">
                  <div className="flex flex-wrap gap-1 max-w-xs">
                    {arr.slice(4).map((tag, idx) => (<Badge key={idx} variant="outline">{tag}</Badge>))}
                  </div>
                </HoverCardContent>
              </HoverCard>
            )
          })}
        </div>
      );
    },
  },
  {
    id: "Times Ran",
    accessorKey: "times_ran",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Times Ran" />
    },
  },
  {
    id: "Created At",
    accessorKey: "created_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Created At" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const createdAtValue = row.getValue("Created At");

      if (!createdAtValue || (typeof createdAtValue !== "string")) {
        return <div>{t('Unknown')}</div>;
      }

      const createdAt = new Date(createdAtValue);
      return (
        <div>
          {createdAt.toLocaleString()}
        </div>
      );
    },
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const { t } = useTranslation();
      const profile = row.original;
      const addProfileMutation = useAddProfile();
      const { user } = useAuth();

      const handleClone = () => {
        const allNames = data.map((p) => p.name);
        let cloneName = `${profile.name} (Clone)`;

        while (allNames.includes(cloneName)) cloneName += " (Clone)";
        addProfileMutation.mutate({
          name: cloneName,
          parameters: profile.parameters,
          description: profile.description,
          tags: profile.tags || [],
        });
      };

      const canPost = hasPermission(user?.role, 'nmap_permissions', 'profiles', 'post');
      const canEdit = hasPermission(user?.role, 'nmap_permissions', 'profiles', 'put');
      const canDelete = hasPermission(user?.role, 'nmap_permissions', 'profiles', 'delete');
      const canScan = hasPermission(user?.role, 'nmap_permissions', 'scans', 'post');
      const canScheduledScan = hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'post');
      const canShowMenu = canPost || canEdit || canDelete || canScan || canScheduledScan;

      return (
        <>
          {canShowMenu &&
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" className="h-8 w-8 p-0">
                  <span className="sr-only">{t('Open menu')}</span>
                  <MoreHorizontal className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="center">
                {canScan &&
                  <DropdownMenuItem onClick={() => handleRunScan(profile)}>
                    <Plus />
                    {t('Run a Scan')}
                  </DropdownMenuItem>
                }
                {canScheduledScan &&
                  <DropdownMenuItem onClick={() => handleAddScheduledScan(profile)}>
                    <CalendarPlus />
                    {t('Create Scheduled Scan')}
                  </DropdownMenuItem>
                }
                {(canScan || canScheduledScan) && (canEdit || canPost || canDelete) && <DropdownMenuSeparator />}
                {canEdit &&
                  <DropdownMenuItem onClick={() => handleEditProfile(profile)}>
                    <Pencil />
                    {t('Edit')}
                  </DropdownMenuItem>
                }
                {canPost &&
                  <DropdownMenuItem onClick={handleClone}>
                    <CopyPlus />
                    {t('Clone')}
                  </DropdownMenuItem>
                }
                {canDelete &&
                  <DropdownMenuItem onClick={() => handleDelete(profile.id)}>
                    <Trash2 />
                    {t('Delete')}
                  </DropdownMenuItem>
                }
              </DropdownMenuContent>
            </DropdownMenu>
          }
        </>
      );
    },
  },
]