import {
  Card,
  CardContent,
} from "@/components/ui/card";
import { useTranslation } from 'react-i18next';
import InfoCard from "@/components/ui/InfoCard";
import { Network, List, Hash, Layers, Volume2, Bug } from "lucide-react";

type ScanInfoProps = {
  jsonResults: any;
  isPending?: boolean;
  isError?: boolean;
};

export default function ScanInfo({ jsonResults, isPending, isError }: ScanInfoProps) {
  const { t } = useTranslation();

  if (isPending) {
    return <Card><CardContent>{t('Loading...')}</CardContent></Card>;
  }

  if (isError || !jsonResults) {
    return <Card><CardContent>{t('Failed to load scan.')}</CardContent></Card>;
  }

  return (
    <Card>
      <CardContent className="space-y-4">
        <div className="flex flex-wrap gap-4">
          {jsonResults.scaninfo && (
            <>

              <div className="w-full grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-4">
                {jsonResults.scaninfo.type && (
                  <InfoCard
                    header={t('Scan Type')}
                    data={jsonResults.scaninfo.type}
                    icon={<Network />}
                    tooltip={t('The type of scan performed (e.g., SYN, Connect, etc.).')}
                  />
                )}
                {jsonResults.scaninfo.protocol && (
                  <InfoCard
                    header={t('Protocol')}
                    data={jsonResults.scaninfo.protocol}
                    icon={<Layers />}
                    tooltip={t('The protocol used for the scan (e.g., TCP, UDP).')}
                  />
                )}
                {jsonResults.scaninfo.numservices && (
                  <InfoCard
                    header={t('Number of Services')}
                    data={jsonResults.scaninfo.numservices}
                    unit={t('ports/services')}
                    icon={<Hash />}
                    tooltip={t('The number of ports/services scanned.')}
                  />
                )}
              </div>

              <div className="w-full">
                {jsonResults.scaninfo.services && (
                  <InfoCard
                    header={t('Services')}
                    data={
                      <div className="gap-2 flex flex-wrap">
                        {jsonResults.scaninfo.services.split(',').map((service: any) => (
                          <div key={service}>{service}</div>
                        ))}
                      </div>
                    }
                    icon={<List />}
                    tooltip={t('The list of ports/services scanned.')}
                    cardContentClassName="text-wrap max-w-full flex flex-wrap gap-2 text-base font-medium"
                    expandable={true}
                  />
                )}
              </div>

              {Object.entries(jsonResults.scaninfo)
                .filter(([key]) => !['type', 'protocol', 'numservices', 'services'].includes(key))
                .map(([key, value]) => (
                  <InfoCard
                    key={key}
                    header={t(key.charAt(0).toUpperCase() + key.slice(1))}
                    data={String(value)}
                    icon={<Layers />}
                    tooltip={t('Scan info field: ') + key}
                  />
                ))}
            </>
          )}

          {jsonResults.verbose &&
            Object.entries(jsonResults.verbose).map(([key, value]) => (
              <InfoCard
                key={"verbose-" + key}
                header={t('Verbosity') + ' ' + key}
                data={String(value)}
                icon={<Volume2 />}
                tooltip={t('Verbosity level for the scan.')}
              />
            ))}
          {jsonResults.debugging &&
            Object.entries(jsonResults.debugging).map(([key, value]) => (
              <InfoCard
                key={"debugging-" + key}
                header={t('Debugging') + ' ' + key}
                data={String(value)}
                icon={<Bug />}
                tooltip={t('Debugging level for the scan.')}
              />
            ))}
        </div>
      </CardContent>
    </Card>
  );
}