import { Button } from "@/components/ui/button"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form'
import { useForm, SubmitHandler } from 'react-hook-form'
import React, { useEffect, useState } from "react"
import { useTranslation } from "react-i18next"
import { useAddScan } from "@/components/nmap/scans/queries";
import { nmapScan } from "@/components/nmap/scans/types";
import { nmapProfile } from "@/components/nmap/profiles/types";
import { validateScanHasInterfaceFlag, validateScanHasTarget, validateScanParameters } from '@/components/nmap/utils';
import { useGetProfiles } from "@/components/nmap/profiles/queries";
import { PageSeparator } from "@/components/Page"
import { Combobox } from "@/components/ui/combobox";

export default function AddNmapScanDialog({ 
  children,
  open: controlledOpen,
  onOpenChange: controlledOnOpenChange,
  parameters = "",
}: { 
  children: React.ReactNode,
  open?: boolean,
  onOpenChange?: (open: boolean) => void,
  parameters?: string,
}) {
  const [open, setOpen] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const { t } = useTranslation();
  const addNmapScanMutation = useAddScan();
  const { data = [] as nmapProfile[] } = useGetProfiles();
  const sortedProfiles = [...data].sort((a, b) => a.name.localeCompare(b.name));

  const form = useForm<Partial<nmapScan>>({
    defaultValues: {
        title: '',
        scan_parameters: '',
        nmap_profile_id: null,
    },
  });

  
  useEffect(() => {
    if (parameters) {
      form.reset({
        title: '',
        scan_parameters: parameters,
        nmap_profile_id: null,
      });
    }
  }, [parameters]);

  const onSubmit: SubmitHandler<Partial<nmapScan>> = (data) => {
    if (!validateScanHasTarget(data.scan_parameters || "", setError)) return;
    if (!validateScanParameters(data.scan_parameters || "", setError)) return;
    if (!validateScanHasInterfaceFlag(data.scan_parameters || "", setError)) return;
    addNmapScanMutation.mutate({
      title: data.title ?? '',
      scan_parameters: data.scan_parameters ?? '',
      nmap_profile_id: data.nmap_profile_id ?? null,
    }, {
      onSuccess: () => {
        form.reset();
        setError(null);
        if (controlledOnOpenChange) {
          controlledOnOpenChange(false);
        } else {
          setOpen(false);
        }
      },
    });
  };

  const handleDialogChange = (isOpen: boolean) => {
    if (controlledOnOpenChange) {
      controlledOnOpenChange(isOpen);
    } else {
      setOpen(isOpen);
    }
    if (!isOpen) {
      setError(null);
      form.reset();
    }
  };

  const handleProfileChange = (profileId: string) => {
    const selectedProfile = data.find((p) => String(p.id) === profileId);
    if (selectedProfile) {
      const existingProfileId = form.getValues("nmap_profile_id");
      let combinedParams = selectedProfile.parameters || "";

      if (!existingProfileId) {
        const currentParams = form.getValues("scan_parameters") || "";
        const profileParams = selectedProfile.parameters || "";
        combinedParams = currentParams
          ? `${currentParams} ${profileParams}`.trim()
          : profileParams;
      }

      form.setValue("title", selectedProfile.name || "");
      form.setValue("scan_parameters", combinedParams);
      form.setValue("nmap_profile_id", selectedProfile.id);
    } else {
      form.setValue("nmap_profile_id", null);
    }
  };

  const isOpen = controlledOpen !== undefined ? controlledOpen : open;
  
  return (
    <Dialog open={isOpen} onOpenChange={handleDialogChange}>
      <DialogTrigger asChild>
        <div>{children}</div>
      </DialogTrigger>
      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle>{t('Add Nmap Scan')}</DialogTitle>
          <DialogDescription>{t('Use this to create a new Nmap Scan')}</DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
            {error && <div className="text-destructive text-sm mb-2">{error}</div>}
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t("Profile")}</FormLabel>
              <div className="col-span-3">
                <Combobox
                  options={sortedProfiles.map((p) => ({ value: String(p.id), label: p.name }))}
                  value={form.watch("nmap_profile_id") ? String(form.watch("nmap_profile_id")) : ""}
                  onChange={handleProfileChange}
                  placeholder={t("Select a profile")}
                  className="w-full"
                />
              </div>
            </FormItem>
            <PageSeparator />
            <FormField
              control={form.control}
              name="title"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Scan Name')}</FormLabel>
                  <FormControl>
                    <Input {...field} className="col-span-3 border-border" />
                  </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="scan_parameters"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="text-right">{t('Scan Parameters')}</FormLabel>
                  <FormControl>
                      <Input {...field} className="col-span-3 border-border" placeholder={t('e.g., -e eth0 -sV -p 80')} />
                  </FormControl>
                </FormItem>
              )}
            />
            <DialogFooter>
              <Button type="submit" className="justify-end">
                  {t('Submit')}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  )
}