"use client"

import { ColumnDef } from "@tanstack/react-table"
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import { nmapScan } from "@/components/nmap/scans/types"
import { Badge } from "@/components/ui/badge"
import { useNavigate } from "react-router-dom"
import { useTranslation } from 'react-i18next';
import { 
  MoreHorizontal, 
  Eye,
  Download,
  RefreshCw,
  Trash2,
  Square,
} from "lucide-react"
import { Button } from "@/components/ui/button"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { 
  useDownloadScan,
  useAddScan,
  useStopScan,
} from "@/components/nmap/scans/queries"
import { useCallback } from "react"
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip"
import { Checkbox } from "@/components/ui/checkbox"
import type { VariantProps } from "class-variance-authority"
import { badgeVariants } from "@/components/ui/badge"
import { Duration } from "@/components/ui/Duration";
import { useAuth } from "@/contexts/AuthContext"
import { hasPermission } from "@/components/role/utils"

type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

export const columns = (
  handleDelete: (id: number) => void,
) : ColumnDef<nmapScan>[] => [
{
  id: "select",
  header: ({ table }) => (
    <Checkbox
      checked={
        table.getIsAllPageRowsSelected() ||
        (table.getIsSomePageRowsSelected() && "indeterminate")
      }
      onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
      aria-label="Select all"
    />
  ),
  cell: ({ row }) => (
    <Checkbox
      checked={row.getIsSelected()}
      onCheckedChange={(value) => row.toggleSelected(!!value)}
      aria-label="Select row"
    />
  ),
  enableSorting: false,
  enableHiding: false,
},
{
  accessorKey: "id",
  header: () => <div className="hidden">ID</div>,
  cell: ({ row }) => (
    <div className="hidden">
      {row.getValue("id")}
    </div>
  ),
  enableSorting: false,
  enableHiding: false,
},
{
  accessorKey: "title",
  header: ({ column }) => {
    return <DataTableColumnHeader column={column} title="Scan Name" />
  },
},
{
  id: "view",
  cell: ({ row }) => {
    const scan = row.original
    const navigate = useNavigate();
    const { t } = useTranslation();

    function viewScan(id: number) {
      navigate(`/nmap/scan/${id}`);
    }

    return (
      scan.status === "Completed" && (
        <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>
            <Button
            variant="ghost"
            className="h-8 w-8 p-0"
            onClick={() => {
              viewScan(scan.id);
            }}
            >
              <Eye />
              <span className="sr-only">{t('View scan results')}</span>
            </Button>
          </TooltipTrigger>
          <TooltipContent side="right">
            <p>{t('View Scan Results')}</p>
          </TooltipContent>
        </Tooltip>
        </TooltipProvider>
      )
    )
  },
},
{
  accessorKey: "status",
  header: ({ column }) => {
    return <DataTableColumnHeader column={column} title="Status" />
  },
  cell: ({ row }) => {
    const status = row.getValue("status")
    let variant: BadgeVariant = "default"

    if (status === "Completed") {
      variant = "green"
    } else if (status === "In Progress" || status === "Pending") {
      variant = "blue"
    } else if (status === "Failed" || status === "Stopped") {
      variant = "red"
    } 
    return <Badge variant={variant}>{String(status)}</Badge>
  },
},
{
  id: "Parameters",
  accessorKey: "scan_parameters",
  header: ({ column }) => {
    return <DataTableColumnHeader column={column} title="Parameters" />
  },
},
{
  accessorKey: "duration",
  header: ({ column }) => <DataTableColumnHeader column={column} title="Duration" />,
  cell: ({ row }) => {
    const scan = row.original;
    const scan_type = scan.scan_type;
    const percent = scan.progress_percent;
    return (
      <div style={{ display: "flex", flexDirection: "column" }}>
        <Duration
          started_at={scan.started_at as string | undefined}
          finished_at={scan.finished_at as string | undefined}
        />
        {typeof percent === "number" && 
        percent !== 100.0 && 
        percent !== 0.0 &&
        (
          <span style={{ fontSize: "0.85em", color: "#888" }}>
            {scan_type + " - " + percent.toFixed(2)}%
          </span>
        )}
      </div>
    );
  },
},
{
  id: "Started At",
  accessorKey: "started_at",
  header: ({ column }) => {
    return <DataTableColumnHeader column={column} title="Started At" />
  },
  cell: ({ row }) => {
    const { t } = useTranslation();
    const startedAtValue = row.getValue("Started At");

    if (!startedAtValue || (typeof startedAtValue !== "string")) {
      return <div>{t('Scan Pending')}...</div>;
    }

    const createdAt = new Date(startedAtValue);
    return (
      <div>
        {createdAt.toLocaleString()}
      </div>
    );
  },
},
{
  id: "Finished At",
  accessorKey: "finished_at",
  header: ({ column }) => {
    return <DataTableColumnHeader column={column} title="Finished At" />
  },
  cell: ({ row }) => {
    const { t } = useTranslation();
    const finishedAtValue = row.getValue("Finished At");
    const startedAtValue = row.getValue("Started At");
    
    if (!finishedAtValue || (typeof finishedAtValue !== "string")) {
      if (startedAtValue) {
        return <div>{t('Scan in progress')}...</div>;
      }
      return <div>{t('Scan Pending')}...</div>;
    }

    const finishedAt = new Date(finishedAtValue);
    return (
      <div>
        {finishedAt.toLocaleString()}
      </div>
    );
  },
},
{
  id: "actions",
  cell: ({ row }) => {
    const { t } = useTranslation();
    const scan = row.original
    const { mutate: downloadScan } = useDownloadScan()
    const addNmapScanMutation = useAddScan();
    const stopScanMutation = useStopScan();
    const { user } = useAuth();

    const handleDownload = useCallback((ext: "xml" | "json" | "txt") => {
      const filename = `${scan.output_filename}.${ext}`
      downloadScan(filename, {
        onSuccess: (data: Blob) => {
          const url = window.URL.createObjectURL(new Blob([data], { type: ext === "xml" ? "application/xml" : "text/plain" }))
          const link = document.createElement("a")
          link.href = url
          link.setAttribute("download", `${scan.output_filename}.${ext}`)
          document.body.appendChild(link)
          link.click()
          link.parentNode?.removeChild(link)
          window.URL.revokeObjectURL(url)
        },
      })
    }, [downloadScan, scan])

    const handleRerunScan = () => {
      addNmapScanMutation.mutate({
        title: scan.title,
        scan_parameters: scan.scan_parameters,
      });
    };

    const handleStopScan = () => {
      stopScanMutation.mutate(scan.id);
    }

    const canStop = hasPermission(user?.role, 'nmap_permissions', 'scans', 'post');
    const canDelete = hasPermission(user?.role, 'nmap_permissions', 'scans', 'delete');

    return (
      <DropdownMenu>
        <DropdownMenuTrigger asChild>
          <Button variant="ghost" className="h-8 w-8 p-0">
            <span className="sr-only">{t('Open menu')}</span>
            <MoreHorizontal className="h-4 w-4" />
          </Button>
        </DropdownMenuTrigger>
        <DropdownMenuContent align="center">
          {canStop &&
            <>
              <DropdownMenuItem onClick={handleRerunScan}>
                <RefreshCw />
                {t('Rerun Scan')}
              </DropdownMenuItem>
              {scan.status == 'In Progress' && (
                <DropdownMenuItem onClick={handleStopScan}>
                  <Square />
                  {t('Stop Scan')}
                </DropdownMenuItem>
              )}
              <DropdownMenuSeparator />
            </>
          }
          <DropdownMenuItem onClick={() => handleDownload("xml")}>
            <Download />
            {t('Download XML')}
          </DropdownMenuItem>
          <DropdownMenuItem onClick={() => handleDownload("json")}>
            <Download />
            {t('Download JSON')}
          </DropdownMenuItem>
          <DropdownMenuItem onClick={() => handleDownload("txt")}>
            <Download />
            {t('Download Text')}
          </DropdownMenuItem>
          {canDelete &&
            <>
              <DropdownMenuSeparator />
              <DropdownMenuItem onClick={() => handleDelete(scan.id)}>
                <Trash2 />
                {t('Delete')}
              </DropdownMenuItem>
            </>
          }
        </DropdownMenuContent>
      </DropdownMenu>
    )
  },
},
]