import { Button } from "@/components/ui/button"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select"
import { Input } from "@/components/ui/input"
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form'
import { useForm, SubmitHandler } from 'react-hook-form'
import React, { useState, isValidElement, useEffect } from "react"
import { useTranslation } from "react-i18next"
import { useAddScheduledScan, useUpdateScheduledScan } from "@/components/nmap/scheduledscans/queries";
import { scheduledNmapScan } from "@/components/nmap/scheduledscans/types";
import { nmapProfile } from "@/components/nmap/profiles/types";
import { useGetProfiles } from "@/components/nmap/profiles/queries";
import { PageSeparator } from "@/components/Page"
import { Checkbox } from "@/components/ui/checkbox"
import { validateScanHasTarget, validateScanParameters, scheduleTypeOptions, validateScanHasInterfaceFlag } from '@/components/nmap/utils';
import { MultiSelect } from "@/components/ui/multi-select";
import { Combobox } from "@/components/ui/combobox";
import { getDayOfMonthWithSuffix } from "@/components/nmap/utils"
import TimePicker from "@/components/ui/TimePicker"

export default function AddEditScheduledScanDialog({
  children,
  scheduledScan,
  scanProfile,
  open: controlledOpen,
  onOpenChange: controlledOnOpenChange,
  mode = "add",
}: {
  children: React.ReactNode,
  scheduledScan?: scheduledNmapScan,
  scanProfile?: nmapProfile,
  open?: boolean,
  onOpenChange?: (open: boolean) => void,
  mode?: "add" | "edit",
}) {
  const [open, setOpen] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const { t } = useTranslation();
  const addScheduledScanMutation = useAddScheduledScan();
  const editScheduledScanMutation = useUpdateScheduledScan();
  const { data = [] } = useGetProfiles();
  const sortedProfiles = [...data].sort((a, b) => a.name.localeCompare(b.name));

  const dayOptions = [
    { value: '0', label: t('Sunday') },
    { value: '1', label: t('Monday') },
    { value: '2', label: t('Tuesday') },
    { value: '3', label: t('Wednesday') },
    { value: '4', label: t('Thursday') },
    { value: '5', label: t('Friday') },
    { value: '6', label: t('Saturday') },
  ];

  const form = useForm<scheduledNmapScan>({
    defaultValues:{
      id: undefined,
      name: '',
      parameters: '',
      nmap_profile_id: null,
      schedule_type: 'everyMinute',
      schedule_parameters: {},
      enabled: true,
    },
  });

  useEffect(() => {
    if (mode === "edit" && scheduledScan) {
      form.reset({
        id: scheduledScan.id,
        name: scheduledScan.name,
        parameters: scheduledScan.parameters,
        nmap_profile_id: scheduledScan.nmap_profile_id ?? null,
        schedule_type: scheduledScan.schedule_type,
        schedule_parameters: (
          scheduledScan.schedule_parameters && !Array.isArray(scheduledScan.schedule_parameters)
            ? scheduledScan.schedule_parameters
            : {}
        ),
        enabled: !!scheduledScan.enabled,
      });
    } else if (mode === "add" && scanProfile) {
      form.reset({
        name: scanProfile.name,
        parameters: scanProfile.parameters,
        nmap_profile_id: scanProfile.id,
        schedule_type: 'everyMinute',
        schedule_parameters: {},
        enabled: true,
      });
    } else {
      form.reset({
        id: undefined,
        name: '',
        parameters: '',
        nmap_profile_id: null,
        schedule_type: 'everyMinute',
        schedule_parameters: {},
        enabled: true,
      });
    }
  }, [scheduledScan, scanProfile]);

  const onSubmit: SubmitHandler<scheduledNmapScan> = (data) => {
    if (!validateScanHasTarget(data.parameters || "", setError)) return;
    if (!validateScanParameters(data.parameters || "", setError)) return;
    if (!validateScanHasInterfaceFlag(data.parameters || "", setError)) return;
    if (mode === "add") {
      addScheduledScanMutation.mutate(data, {
        onSuccess: () => {
          form.reset();
          setError(null);
          if (controlledOnOpenChange) {
            controlledOnOpenChange(false);
          } else {
            setOpen(false);
          }
        },
      });
    } else {
      const id = scheduledScan?.id;
      if (!id) return;
      const { user_id, id: _id, ...updates } = data;

      if (updates.schedule_parameters) {
        const cleanedParameters: Record<string, any> = {};
        Object.entries(updates.schedule_parameters).forEach(([key, value]) => {
          if (Array.isArray(value)) {
            const nonNullValues = value.filter((v) => v !== null && v !== '');
            if (nonNullValues.length > 0) {
              cleanedParameters[key] = value;
            }
          } else {
            if (value !== null && value !== '') {
              cleanedParameters[key] = value;
            }
          }
        });
        updates.schedule_parameters = cleanedParameters;
      }

      editScheduledScanMutation.mutate({
        id,
        updates,
      }, {
        onSuccess: () => {
          form.reset();
          setError(null);
          if (controlledOnOpenChange) {
            controlledOnOpenChange(false);
          } else {
            setOpen(false);
          }
        },
      });
    }
  };

  const handleDialogChange = (isOpen: boolean) => {
    if (controlledOnOpenChange) {
      controlledOnOpenChange(isOpen);
    } else {
      setOpen(isOpen);
    }
    if (!isOpen) {
      setError(null);
      form.reset();
    }
  };

  const handleProfileChange = (profileId: string) => {
    const selectedProfile = data.find((p) => String(p.id) === profileId);
    if (selectedProfile) {
      form.setValue("name", selectedProfile.name || "");
      form.setValue("parameters", selectedProfile.parameters || "");
      form.setValue("nmap_profile_id", selectedProfile.id);
      setError(null);
    } else {
      form.setValue("nmap_profile_id", null);
    }
  };
  
  const isOpen = controlledOpen !== undefined ? controlledOpen : open;
  const selectedType = scheduleTypeOptions.find(opt => opt.value === form.watch('schedule_type'));
  
  return (
    <Dialog open={isOpen} onOpenChange={handleDialogChange}>
      {controlledOpen === undefined && (
        <DialogTrigger asChild>
          {isValidElement(children) ? children : <span>{children}</span>}
        </DialogTrigger>
      )}
      <DialogContent className="sm:max-w-[830px]">
        <DialogHeader>
          <DialogTitle>{mode === "edit" ? t('Edit Scheduled Scan') : t('Add Scheduled Scan')}</DialogTitle>
          <DialogDescription>{mode === "edit" ? t('Update the details for this scheduled scan.') : t('Use this to create a new Scheduled Scan')}</DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
          {error && <div className="text-destructive text-sm mb-2">{error}</div>}
          <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t("Profile")}</FormLabel>
              <div className="col-span-3">
                <Combobox
                  options={sortedProfiles.map((p) => ({ value: String(p.id), label: p.name }))}
                  value={form.watch("nmap_profile_id") ? String(form.watch("nmap_profile_id")) : ""}
                  onChange={handleProfileChange}
                  placeholder={t("Select a profile")}
                  className="w-full"
                />
              </div>
            </FormItem>
            <FormField
              control={form.control}
              name="name"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Name')}</FormLabel>
                  <FormControl>
                    <Input {...field} className="col-span-3 border-border" />
                  </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="parameters"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Scan Parameters')}</FormLabel>
                  <FormControl>
                      <Input {...field} className="col-span-3 border-border" placeholder={t('e.g., -sV -p 80')} />
                  </FormControl>
                </FormItem>
              )}
            />
            <PageSeparator />
            <FormField
              control={form.control}
              name="schedule_type"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Schedule Type')}</FormLabel>
                  <FormControl>
                    <Select
                      value={field.value}
                      onValueChange={(val) => {
                        field.onChange(val);
                        const allArgFields = [
                          'minute', 'hour1', 'hour2', 'weekday', 'day', 'day1', 'day2', 'month', 'time', 'expression'
                        ];
                        allArgFields.forEach(arg => form.setValue(`schedule_parameters.${arg}`, undefined));
                      }}
                    >
                      <SelectTrigger className="col-span-3 w-full">
                        <SelectValue placeholder={t('Select schedule type')} />
                      </SelectTrigger>
                      <SelectContent className="max-h-80">
                        <SelectGroup>
                          {scheduleTypeOptions.map((opt) => (
                            <SelectItem key={opt.value} value={opt.value}>
                              {t(opt.label)}
                            </SelectItem>
                          ))}
                        </SelectGroup>
                      </SelectContent>
                    </Select>
                  </FormControl>
                </FormItem>
              )}
            />
            {selectedType?.fields && selectedType.fields.length > 0 && (
              <>
                {selectedType.fields.map(f => {
                  if (f.name === 'weekday') {
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl>
                          <Select
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                          >
                            <SelectTrigger className="col-span-3 w-full">
                              <SelectValue placeholder={t('Select weekday')} />
                            </SelectTrigger>
                            <SelectContent className="max-h-80">
                              <SelectGroup>
                                {dayOptions.map(opt => (
                                  <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                                ))}
                              </SelectGroup>
                            </SelectContent>
                          </Select>
                        </FormControl>
                      </FormItem>
                    );
                  } else if (f.name === 'month') {
                    const monthOptions = [
                      { value: '1', label: t('January') },
                      { value: '2', label: t('February') },
                      { value: '3', label: t('March') },
                      { value: '4', label: t('April') },
                      { value: '5', label: t('May') },
                      { value: '6', label: t('June') },
                      { value: '7', label: t('July') },
                      { value: '8', label: t('August') },
                      { value: '9', label: t('September') },
                      { value: '10', label: t('October') },
                      { value: '11', label: t('November') },
                      { value: '12', label: t('December') },
                    ];
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl>
                          <Select
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                          >
                            <SelectTrigger className="col-span-3 w-full">
                              <SelectValue placeholder={t('Select month')} />
                            </SelectTrigger>
                            <SelectContent className="max-h-80">
                              <SelectGroup>
                                {monthOptions.map(opt => (
                                  <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                                ))}
                              </SelectGroup>
                            </SelectContent>
                          </Select>
                        </FormControl>
                      </FormItem>
                    );
                  } else if (['Day', 'Day 1', 'Day 2'].includes(f.label)) {
                    const dayNumOptions = Array.from({ length: 31 }, (_, i) => ({ value: String(i + 1), label: String(i + 1) }));
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl>
                          <Select
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                          >
                            <SelectTrigger className="col-span-3 w-full">
                              <SelectValue placeholder={t('Select day')} />
                            </SelectTrigger>
                            <SelectContent className="max-h-80">
                              <SelectGroup>
                                {dayNumOptions.map(opt => (
                                  <SelectItem key={opt.value} value={opt.value}>{getDayOfMonthWithSuffix(opt.label)}</SelectItem>
                                ))}
                              </SelectGroup>
                            </SelectContent>
                          </Select>
                        </FormControl>
                      </FormItem>
                    );
                  } else if (['hour1', 'hour2'].includes(f.name)) {
                    const hourOptions = Array.from({ length: 24 }, (_, i) => ({ value: String(i), label: String(i).padStart(2, '0') }));
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl>
                          <Select
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                          >
                            <SelectTrigger className="col-span-3 w-full">
                              <SelectValue placeholder={t('Select hour')} />
                            </SelectTrigger>
                            <SelectContent className="max-h-80">
                              <SelectGroup>
                                {hourOptions.map(opt => (
                                  <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                                ))}
                              </SelectGroup>
                            </SelectContent>
                          </Select>
                        </FormControl>
                      </FormItem>
                    );
                  } else if (f.type === 'time') {
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl className="col-span-3">
                          <TimePicker
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onChange={(val: string) => form.setValue(`schedule_parameters.${f.name}`, val)}
                            className="w-full"
                          />
                        </FormControl>
                      </FormItem>
                    );
                  } else {
                    return (
                      <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                        <FormLabel>{t(f.label)}</FormLabel>
                        <FormControl>
                          <Input
                            type={f.type === 'number' ? 'number' : 'text'}
                            value={form.watch(`schedule_parameters.${f.name}`) || ''}
                            onChange={e => form.setValue(`schedule_parameters.${f.name}`, e.target.value)}
                            className="col-span-3 border-border"
                            placeholder={f.label}
                          />
                        </FormControl>
                      </FormItem>
                    );
                  }
                })}
              </>
            )} 
            {selectedType?.constraints && (
              <>
                <FormField
                  control={form.control}
                  name="schedule_parameters.days"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                      <FormLabel>{t('Days')}</FormLabel>
                      <FormControl className="col-span-3 w-full">
                        <MultiSelect
                          options={dayOptions}
                          onValueChange={field.onChange}
                          defaultValue={field.value || []}
                          placeholder={t("Select days")}
                          maxCount={7}
                          useSearch={false}
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="schedule_parameters.between"
                  render={({ field }) => (
                    <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                      <FormLabel>{t('Between')}</FormLabel>
                      <FormControl>
                        <TimePicker 
                          mode="range"
                          value={field.value || ['', '']}
                          onChange={field.onChange}
                        />
                      </FormControl>
                    </FormItem>
                  )}
                />
              </>
            )}
            <PageSeparator />
            <FormField
              control={form.control}
              name="enabled"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="mb-0 cursor-pointer" onClick={() => field.onChange(!field.value)}>
                    {t('Enabled')}
                  </FormLabel>
                  <FormControl>
                    <Checkbox
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                  </FormControl>
                </FormItem>
              )}
            />
            <DialogFooter>
              <Button type="submit" className="justify-end">
                  {t('Submit')}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  )
}