import { Button } from "@/components/ui/button";
import { DataTable, DataTableRowAction } from "@/components/ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import React from 'react';
import { columns } from "@/components/nmap/scheduledscans/columns";
import AddEditScheduledScanDialog from "@/components/nmap/scheduledscans/AddEditScheduledScanDialog";
import DeleteScheduledScanDialog from "@/components/nmap/scheduledscans/DeleteScheduledScanDialog";
import { scheduledNmapScan } from "@/components/nmap/scheduledscans/types";
import { Plus } from "lucide-react";
import { useTranslation } from 'react-i18next';
import { useAuth } from "@/contexts/AuthContext";
import { hasPermission } from "@/components/role/utils";

type ScheduledScansProps = {
  data: scheduledNmapScan[] | undefined;
  isLoading: boolean;
}

export default function ScheduledScans({ data, isLoading }: ScheduledScansProps) {
  const { t } = useTranslation();
  const [editDialogOpen, setEditDialogOpen] = React.useState(false);
  const [scheduledScanToEdit, setScheduledScanToEdit] = React.useState<scheduledNmapScan | undefined>(undefined);
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [idsToDelete, setIdsToDelete] = React.useState<number[]>([]);
  const { user } = useAuth();

  const handleAdd = () => {
    setScheduledScanToEdit(undefined);
    setEditDialogOpen(true);
  }

  const handleEdit = (scheduledScan: scheduledNmapScan) => {
    setScheduledScanToEdit(scheduledScan);
    setEditDialogOpen(true);
  }

  const handleDelete = (id: number) => {
    setIdsToDelete([id]);
    setDeleteDialogOpen(true);
  }

  const actions: DataTableRowAction<scheduledNmapScan>[] = [];

  if (hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'delete')) {
    actions.push(
      {
        label: t("Delete"),
        value: "delete",
        onClick: async (selectedRows: any) => {
          setIdsToDelete(selectedRows.map((row: any) => row.id));
          setDeleteDialogOpen(true);
        }
      }
    );
  }

  return (
    <>
      {hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'post') &&
        <div className="pb-1 flex">
          <Button onClick={handleAdd}>
            <Plus />
            {t('Schedule a Scan')}
          </Button>
        </div>
      }
      {!isLoading && (
        <DataTable 
          columns={columns(
            handleEdit,
            handleDelete,
          )} 
          data={data || []} 
          sorting={[{ id: "Last Ran", desc: true }]}
          usesRowSelection={true}
          selectedRowActions={actions}
          defaultColumnVisibility={{
            'Duration': false,
            'Finished At': false,
            'Created At': false,
            'Times Ran': false,
          }}
        />
      )}
      {isLoading && <DataTableSkeleton />}
      <AddEditScheduledScanDialog
        open={editDialogOpen}
        onOpenChange={setEditDialogOpen}
        scheduledScan={scheduledScanToEdit}
        mode={scheduledScanToEdit ? "edit" : "add"}
      >
        <></>
      </AddEditScheduledScanDialog>
      <DeleteScheduledScanDialog
        ids={idsToDelete}
        open={deleteDialogOpen}
        onOpenChange={(open) => {
          setDeleteDialogOpen(open);
          if (!open) setIdsToDelete([]);
        }}
      >
        <span />
      </DeleteScheduledScanDialog>
    </>
  )
}