import { ColumnDef, Row } from "@tanstack/react-table"
import { Badge } from "@/components/ui/badge"
import { Checkbox } from "@/components/ui/checkbox"
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader"
import type { VariantProps } from "class-variance-authority"
import { badgeVariants } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { useTranslation } from 'react-i18next';
import { 
  MoreHorizontal,
  Pencil,
  Trash2,
} from "lucide-react"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { useState, useEffect } from "react"
import { scheduledNmapScan } from "@/components/nmap/scheduledscans/types"
import { Switch } from "@/components/ui/switch"
import { useUpdateScheduledScan } from "@/components/nmap/scheduledscans/queries"
import { getScheduleTypeLabel, getDayOfMonthWithSuffix } from "@/components/nmap/utils"
import { useAuth } from "@/contexts/AuthContext"
import { hasPermission } from "@/components/role/utils"

type BadgeVariant = VariantProps<typeof badgeVariants>["variant"]

export const columns = (
  handleEdit: (scheduledScan: scheduledNmapScan) => void,
  handleDelete: (id: number) => void,
) : ColumnDef<scheduledNmapScan>[] => [
  {
    id: "select",
    header: ({ table }) => (
      <Checkbox
        checked={
          table.getIsAllPageRowsSelected() ||
          (table.getIsSomePageRowsSelected() && "indeterminate")
        }
        onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
        aria-label="Select all"
      />
      ),
      cell: ({ row }) => (
        <Checkbox
          checked={row.getIsSelected()}
          onCheckedChange={(value) => row.toggleSelected(!!value)}
          aria-label="Select row"
        />
      ),
      enableSorting: false,
      enableHiding: false,
  },
  {
    accessorKey: "name",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scheduled Scan Name" />
    },
  },
  {
    id: "Status",
    accessorKey: "enabled",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Status" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      let variant: BadgeVariant = row.original.enabled ? "green" : "red";
      return (
        <Badge variant={variant}>
          {row.original.enabled ? t("Enabled") : t("Disabled")}
        </Badge>
      );
    },
  },
  {
    id: "Scan Parameters",
    accessorKey: "parameters",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Scan Parameters" />
    },
    cell: ({ row }) => {
      const params = row.getValue("Scan Parameters") as string;
      return (
        <div className="max-w-[200px] whitespace-normal break-words">
          {params}
        </div>
      );
    },
  },  
  {
    id: "Frequency",
    accessorKey: "schedule_type",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Frequency" />
    },
    cell: ({ row }) => {
      const type = row.original.schedule_type;
      const args = row.original.schedule_parameters?.args;
      return (
        <span>
          {getScheduleTypeLabel(type)}
          {args ? ` ${args}` : ""}
        </span>
      );
    },
    enableSorting: true,
    sortingFn: (rowA, rowB) => {
      const labelA = getScheduleTypeLabel(rowA.original.schedule_type);
      const labelB = getScheduleTypeLabel(rowB.original.schedule_type);
      if (labelA < labelB) return -1;
      if (labelA > labelB) return 1;
      return 0;
    },
  },
  {
    id: "Schedule Parameters",
    accessorKey: "schedule_parameters",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Schedule Parameters" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const params = row.original.schedule_parameters || {};

      if (!params || Object.keys(params).length === 0) return <span>-</span>;

      const dayNames = [
        t('Sunday'), t('Monday'), t('Tuesday'), t('Wednesday'), t('Thursday'), t('Friday'), t('Saturday')
      ];
      const monthNames = [
        t('January'), t('February'), t('March'), t('April'), t('May'), t('June'),
        t('July'), t('August'), t('September'), t('October'), t('November'), t('December')
      ];
      
      function formatTime(timeStr: string) {
        if (!timeStr) return "";
        const [hour, minute] = timeStr.split(":");
        let h = typeof hour === "string" ? parseInt(hour, 10) : hour;
        let m = typeof minute === "string" ? parseInt(minute, 10) : minute;
        if (isNaN(h)) return hour;
        const suffix = h >= 12 ? "PM" : "AM";
        let hour12 = h % 12;
        if (hour12 === 0) hour12 = 12;
        let minStr = typeof m === "number" && !isNaN(m) ? `:${m.toString().padStart(2, "0")}` : "";
        return `${hour12}${minStr} ${suffix}`;
      }

      return (
        <div className="max-w-[300px] whitespace-pre-line break-words">
          {params.expression ? (
            <>
              <strong>{t('Expression')}: </strong> 
              {getDayOfMonthWithSuffix(params.expression)}
              <br />
            </>
          ) : ""}
          {params.days ? (
            <>
              <strong>{t('Days of Week')}: </strong>
              {params.days
                .map((index: number) =>
                  index >= 0 && index < dayNames.length ? dayNames[index] : index
                )
                .join(', ')
              }
              <br />
            </>
          ) : ""}
          {params.weekday ? (
            <>
              <strong>{t('Day of Week')}: </strong>
              {(() => {
                let index = params.weekday;
                if (typeof index === 'string') index = parseInt(index, 10);
                return !isNaN(index) && index >= 0 && index < dayNames.length ? dayNames[index] : params.weekday;
              })()}
              <br />
            </>
          ) : ""}
          {params.month ? (
            <>
              <strong>{t('Month')}: </strong> 
              {monthNames[Number(params.month) - 1] || params.month}
              <br />
            </>
          ) : ""}
          {params.day ? (
            <>
              <strong>{t('Day')}: </strong> 
              {getDayOfMonthWithSuffix(params.day)}
              <br />
            </>
          ) : ""}
          {params.day1 && params.day2 ? (
            <>
              <strong>{t('Days')}: </strong>
              {getDayOfMonthWithSuffix(params.day1)} {t('and')} {getDayOfMonthWithSuffix(params.day2)}
              <br />
            </>
          ) : params.day1 ? (
            <>
              <strong>{t('Day')}: </strong>
              {getDayOfMonthWithSuffix(params.day1)}
              <br />
            </>
          ) : params.day2 ? (
            <>
              <strong>{t('Day')}: </strong>
              {getDayOfMonthWithSuffix(params.day2)}
              <br />
            </>
          ) : ""}
         {params.between ? (
            <>
              <strong>{t('Between')}: </strong>
              {(() => {
                const [start, end] = Array.isArray(params.between)
                  ? params.between
                  : params.between.split(/[\s\-–—]+/);
                return (
                  <>
                    {formatTime(start)} {t('and')} {formatTime(end)}
                  </>
                );
              })()}
              <br />
            </>
          ) : ""}
          {params.hour1 && params.hour2 ? (
            <>
              <strong>{t('Hours')}: </strong>
              {formatTime(params.hour1)} {t('and')} {formatTime(params.hour2)}
              <br />
            </>
          ) : params.hour1 ? (
            <>
              <strong>{t('Hour')}: </strong>
              {formatTime(params.hour1)}
              <br />
            </>
          ) : params.hour2 ? (
            <>
              <strong>{t('Hour')}: </strong>
              {formatTime(params.hour2)}
              <br />
            </>
          ) : ""}
          {params.time ? (
            <>
              <strong>{t('Time')}: </strong>
              {formatTime(params.time)}
              <br />
            </>
          ) : ""}
          {params.minute ? (
            <>
              <strong>{t('Minute')}: </strong>
              {params.minute}
              <br />
            </>
          ) : ""}
        </div>
      );
    },
    enableSorting: false,
  },
  {
    id: "Last Ran",
    accessorKey: "last_run_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Last Ran" />
    },
    cell: ({ row }) => {
      let lastRun = row.original.last_run_at;
      if (!lastRun) return <span>Never</span>;
      const date = new Date(lastRun);
      if (isNaN(date.getTime())) return <span>Invalid date</span>;
      return <span>{date.toLocaleString()}</span>;
    },
    sortingFn: (rowA: Row<scheduledNmapScan>, rowB: Row<scheduledNmapScan>) => {
      function getSortableValue(row: Row<scheduledNmapScan>) {
        let value = row.original.last_run_at;
        if (!value) return -Infinity;
        const date = new Date(value);
        if (isNaN(date.getTime())) return -Infinity;
        return date.getTime();
      }
      const a = getSortableValue(rowA);
      const b = getSortableValue(rowB);
      if (a < b) return -1;
      if (a > b) return 1;
      return 0;
    },
  },
  {
    id: "Times Ran",
    accessorKey: "times_ran",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Times Ran" />
    },
    cell: ({ row }) => {
      let timesRan = row.original.times_ran;
      return <span>{timesRan}</span>;
    },
  },
  {
    id: "Created At",
    accessorKey: "created_at",
    header: ({ column }) => {
      return <DataTableColumnHeader column={column} title="Created At" />
    },
    cell: ({ row }) => {
      const { t } = useTranslation();
      const createdAtValue = row.getValue("Created At");
      
      if (!createdAtValue || (typeof createdAtValue !== "string")) {
        return <div>{t('Scan Pending')}...</div>;
      }

      const createdAt = new Date(createdAtValue);
      return (
        <div>
          {createdAt.toLocaleString()}
        </div>
      );
    },
  },
  {
    id: "toggle-enabled",
    header: () => null,
    cell: ({ row }) => {
      const scheduledscan = row.original
      const [switchLoading, setSwitchLoading] = useState(false)
      const [checked, setChecked] = useState(scheduledscan.enabled)
      const updateScheduledScan = useUpdateScheduledScan();
      const { user } = useAuth();

      useEffect(() => {
        setChecked(scheduledscan.enabled)
      }, [scheduledscan.enabled])

      const handleToggle = async () => {
        const newChecked = !checked
        setChecked(newChecked)
        setSwitchLoading(true)
        updateScheduledScan.mutate({
          id: scheduledscan.id,
          updates: { enabled: newChecked },
        }, {
          onError: () => {
            setChecked(scheduledscan.enabled)
          },
          onSettled: () => setSwitchLoading(false)
        })
      }

      return (
        <Switch
          checked={checked}
          disabled={!hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'put') || switchLoading}
          onCheckedChange={handleToggle}
        />
      )
    },
  },
  {
    id: "actions",
    cell: ({ row }) => {
      const { t } = useTranslation();
      const { user } = useAuth();
      const scheduledscan = row.original;
      const canEdit = hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'put');
      const canDelete = hasPermission(user?.role, 'nmap_permissions', 'scheduled_scans', 'delete')
      const canShowMenu = canEdit || canDelete;

      return (
        <>
          {canShowMenu &&
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" className="h-8 w-8 p-0">
                  <span className="sr-only">{t('Open menu')}</span>
                  <MoreHorizontal className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="center">
                {canEdit &&
                  <DropdownMenuItem onClick={() => handleEdit(scheduledscan)}>
                    <Pencil />
                    {t('Edit')}
                  </DropdownMenuItem>
                }
                {canDelete &&
                  <DropdownMenuItem onClick={() => handleDelete(scheduledscan.id)}>
                    <Trash2 />
                    {t('Delete')}
                  </DropdownMenuItem>
                }
              </DropdownMenuContent>
            </DropdownMenu>
          }
        </>
      )
    },
  },
]