import { scheduledNmapScan } from '@/components/nmap/scheduledscans/types';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { toast } from "sonner";

export function useGetScheduledScans() {
  return useQuery<scheduledNmapScan[], AxiosError>({
    queryKey: ['scheduledNmapScans'],
    queryFn: async () => {
      const { data } = await axios.get('/api/v1/nmap/scheduled-scans');
      return data;
    },
    staleTime: 5 * 60 * 1000, // 5 minutes
  });
}

export function useGetScheduledScan(id: string) {
  return useQuery<scheduledNmapScan, AxiosError>({
    queryKey: ['scheduledNmapScan', id],
    queryFn: async () => {
      const { data } = await axios.get(`/api/v1/nmap/scheduled-scans/${id}`);
      return data;
    },
    enabled: !!id,
  });
}

export function useAddScheduledScan() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (scan: Omit<scheduledNmapScan, 'id' | 'user_id'> & { user_id?: number }) =>
      axios.post('/api/v1/nmap/scheduled-scans', scan),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['scheduledNmapScans'] });
      toast(t('Scheduled scan created successfully.'));
    },
    onError: () => {
      toast(t('Failed to create scheduled scan.'), {
        description: t('Please try again.'),
      });
    },
  });
}

export function useUpdateScheduledScan() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, updates }: { id: number; updates: Partial<scheduledNmapScan> }) =>
      axios.put(`/api/v1/nmap/scheduled-scans/${id}`, updates),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['scheduledNmapScans'] });
      toast(t('Scheduled scan updated.'));
    },
    onError: () => {
      toast(t('Failed to update scheduled scan.'), {
        description: t('Please try again.'),
      });
    },
  });
}

export function useDeleteScheduledScans() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (ids: number[]) =>
      axios.delete('/api/v1/nmap/scheduled-scans', {
        data: { ids },
      }),
    onMutate: async (ids) => {
      const key = ['scheduledNmapScans'];
      await queryClient.cancelQueries({ queryKey: key });
      const previous = queryClient.getQueryData<scheduledNmapScan[]>(key);
      queryClient.setQueryData(key, (old: scheduledNmapScan[] = []) =>
        old.filter((s) => !ids.includes(s.id))
      );
      return { previous };
    },
    onError: (_err, ids, context) => {
      queryClient.setQueryData(['scheduledNmapScans'], context?.previous);
      const errorMessage = ids.length > 1 
        ? t('Failed to delete scheduled scans.')
        : t('Failed to delete scheduled scan.');
      toast(errorMessage, {
        description: t('Please try again.'),
      });
    },
    onSuccess: (_data, ids) => {
      const msg = ids.length > 1 
        ? t('Scheduled scans deleted.') 
        : t('Scheduled scan deleted.');
      toast(msg);
    },
    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['scheduledNmapScans'] });
    },
  });
}
