import { nmapScan } from '@/components/nmap/scans/types';
import { Host } from '@/components/nmap/scan/types';

export const forbiddenScanParameterPatterns = [/-oN/, /-oX/, /-oS/, /-oG/, /-oA/];

// 192.168.1.0/24, etc.
const ipAddressWithCIDR = /\b(\d{1,3}\.){3}\d{1,3}\/\d{1,2}\b/;

// 192.168.1.0-255, 192.0-168.0-255.0, etc.
const ipAddressWithRanges = /\b(?:(?:\d{1,3}(?:-\d{1,3})?)\.){3}\d{1,3}(?:-\d{1,3})?\b/;

// nagios.com, libary.nagios.com, nagios.com/24, etc.
const hostNameWithCIDR = /\b((?:[a-zA-Z0-9-]+\.)+[a-zA-Z]{2,})(?:\/\d{1,2})?\b/;

// Gets targets and hosts up from "# Nmap done at Thu Jul  3 14:24:15 2025 -- 256 IP addresses (67 hosts up) scanned in 56.16 seconds" line at end of scan text output
const targetCount = /(\d+)\s+IP address(?:es)?/;
const hostsUpCount = /\((\d+)\s+hosts?\s+up\)/;

export function getIPv4FromHost(host: Host | null): string {
  if (!host) return 'Unknown';
  if (Array.isArray(host.address)) {
    return host.address.find(a => a.addrtype === "ipv4")?.addr ?? 'Unknown';
  }
  if (host.address?.addrtype === "ipv4") {
    return host.address.addr ?? 'Unknown';
  }
  return 'Unknown';
}

export function getMACFromHost(host: Host | null): string {
  if (!host) return 'Unknown';
  if (Array.isArray(host.address)) {
    return host.address.find(a => a.addrtype === "mac")?.addr ?? 'Unknown';
  }
  if (host.address?.addrtype === "mac") {
    return host.address.addr ?? 'Unknown';
  }
  return 'Unknown';
}

export function getScanTargetCount(scan: nmapScan): number {
  const match = scan.scan_results?.match(targetCount);
  return match ? parseInt(match[1], 10) : 0;
}

export function getScanHostsUpCount(scan: nmapScan): number {
  const match = scan.scan_results?.match(hostsUpCount);
  return match ? parseInt(match[1], 10) : 0;
}

export function validateScanParameters(
  params: string,
  setError: (msg: string | null) => void
): boolean {
  const foundPattern = forbiddenScanParameterPatterns.find(pattern => pattern.test(params));
  if (foundPattern) {
    setError('Output file parameters (-oN, -oX, -oS, -oG, -oA) are not allowed.');
    return false;
  }
  setError(null);
  return true;
}

export function validateScanHasInterfaceFlag(
  params: string,
  setError: (msg: string | null) => void,
  profilePage?: boolean
): boolean {
  const regexInterfaceFlag = /.*(-e\s+[a-zA-Z0-9._:@-]+).*/; // checks for -e flag and stores that plus its argument 
  const ipOnly = /\d{1,3}(?:\.\d{1,3}){3}$/; // checks for a simple ip address
  const ipWithCidr = /\d{1,3}(?:\.\d{1,3}){3}\/\d{1,2}$/; // checks for an ip address with cidr notation e.g, 192.168.0.1/24
  const ipRange = /\d{1,3}(?:-\d{1,3})?(?:\.\d{1,3}(?:-\d{1,3})?){3}$/; // checks for an ip address with ranges e.g, 192.168.0.1-5
  // this checks for ipv6 addresses 
  const ipv6Expresion = /(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))/;
  const interfaceflag = regexInterfaceFlag.exec(params);
  if (!interfaceflag) {
    if (profilePage) {
      setError("Please include an interface using -e <interface> (e.g., eth0). Found under \"Firewall Evasion and Spoofing.\"");
      return false;
    }
    setError("Please include an interface using -e <interface> (e.g., eth0)");
    return false;
  } else if (ipOnly.test(interfaceflag?.[1] || '') || ipWithCidr.test(interfaceflag?.[1] || '') || ipRange.test(interfaceflag?.[1] || '') || ipv6Expresion.test(interfaceflag?.[1] || '')) {
    setError("The value provided for the interface flag (-e) appears to be an IP address. Please provide a valid network interface (e.g., eth0).");
    return false;
  }
  setError(null);
  return true;
}

// Validate that the scan has a valid target (IP, hostname, or known flags)
export function validateScanHasTarget(
  params: string,
  setError?: (msg: string | null) => void
): boolean {
  let hasTarget = false;
  
  if (/\b-iL\b|\b-iR\b|--exclude\b|--excludefile\b/.test(params)) {
    hasTarget = true;
  }
  if (ipAddressWithCIDR.test(params)) {
    hasTarget = true;
  }
  if (ipAddressWithRanges.test(params)) {
    hasTarget = true;
  }
  if (hostNameWithCIDR.test(params)) {
    hasTarget = true;
  }

  if (!hasTarget && setError) {
    setError("Specify a scan target in the scan parameters.");
  }

  return hasTarget;
}

export const scheduleTypeOptions = [
  { 
    value: 'everyMinute', 
    label: 'Every Minute', 
    constraints: true, 
  },
  { 
    value: 'everyTwoMinutes', 
    label: 'Every Two Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyThreeMinutes', 
    label: 'Every Three Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyFourMinutes', 
    label: 'Every Four Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyFiveMinutes', 
    label: 'Every Five Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyTenMinutes', 
    label: 'Every Ten Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyFifteenMinutes', 
    label: 'Every Fifteen Minutes', 
    constraints: true, 
  },
  { 
    value: 'everyThirtyMinutes', 
    label: 'Every Thirty Minutes', 
    constraints: true, 
  },
  { 
    value: 'hourlyAt', 
    label: 'Hourly', 
    constraints: true, 
    fields: [ { name: 'minute', label: 'Minute', type: 'number' } ] 
  },
  { 
    value: 'everyOddHour', 
    label: 'Every Odd Hour', 
    constraints: true, 
    fields: [ { name: 'minute', label: 'Minute (optional)', type: 'number', optional: true } ] 
  },
  { 
    value: 'everyTwoHours', 
    label: 'Every Two Hours', 
    constraints: true,
    fields: [ { name: 'minute', label: 'Minute (optional)', type: 'number', optional: true } ] 
  },
  { 
    value: 'everyThreeHours', 
    label: 'Every Three Hours', 
    constraints: true,
    fields: [ { name: 'minute', label: 'Minute (optional)', type: 'number', optional: true } ] 
  },
  { 
    value: 'everyFourHours', 
    label: 'Every Four Hours', 
    constraints: true,
    fields: [ { name: 'minute', label: 'Minute (optional)', type: 'number', optional: true } ] 
  },
  { 
    value: 'everySixHours', 
    label: 'Every Six Hours', 
    constraints: true,
    fields: [ { name: 'minute', label: 'Minute (optional)', type: 'number', optional: true } ] 
  },
  { 
    value: 'dailyAt', 
    label: 'Daily',  
    fields: [ { name: 'time', label: 'Time (HH:MM)', type: 'time' } ] 
  },
  { 
    value: 'twiceDailyAt', 
    label: 'Twice Daily', 
    fields: [ 
      { name: 'hour1', label: 'Hour 1', type: 'number' }, 
      { name: 'hour2', label: 'Hour 2', type: 'number' },
      { name: 'minute', label: 'Minute', type: 'number' } 
    ] 
  },
  { 
    value: 'weeklyOn', 
    label: 'Weekly', 
    fields: [ 
      { name: 'weekday', label: 'Weekday', type: 'number' }, 
      { name: 'time', label: 'Time (HH:MM)', type: 'time' } 
    ] 
  },
  { 
    value: 'monthlyOn', 
    label: 'Monthly', 
    fields: [ 
      { name: 'day', label: 'Day', type: 'number' }, 
      { name: 'time', label: 'Time (HH:MM)', type: 'time' } 
    ] 
  },
  { 
    value: 'twiceMonthly', 
    label: 'Twice Monthly', 
    fields: [ 
      { name: 'day1', label: 'Day 1', type: 'number' }, 
      { name: 'day2', label: 'Day 2', type: 'number' }, 
      { name: 'time', label: 'Time (HH:MM)', type: 'time' } 
    ] 
  },
  { 
    value: 'lastDayOfMonth', 
    label: 'Last Day Of Month', 
    fields: [ { name: 'time', label: 'Time (HH:MM)', type: 'time' } ] 
  },
  { 
    value: 'quarterly', 
    label: 'Quarterly'
  },
  { 
    value: 'yearlyOn', 
    label: 'Yearly', 
    fields: [ 
      { name: 'month', label: 'Month', type: 'number' }, 
      { name: 'day', label: 'Day', type: 'number' }, 
      { name: 'time', label: 'Time (HH:MM)', type: 'time' } 
    ] 
  },
  { 
    value: 'cron', 
    label: 'Custom Cron Expression', 
    fields: [ { name: 'expression', label: 'Cron Expression', type: 'text' } ] 
  },
];

export function getScheduleTypeLabel(type: string): string {
  const found = scheduleTypeOptions.find(opt => opt.value === type);
  return found ? found.label : type;
}

export function getDayOfMonthWithSuffix(day: number | string): string {
  const n = typeof day === 'string' ? parseInt(day, 10) : day;
  if (isNaN(n)) return String(day);
  const j = n % 10, k = n % 100;
  if (j === 1 && k !== 11) return `${n}st`;
  if (j === 2 && k !== 12) return `${n}nd`;
  if (j === 3 && k !== 13) return `${n}rd`;
  return `${n}th`;
}
