import { Report } from "@/components/report/types";
import { Button } from "@/components/ui/button"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Form, FormLabel, FormControl, FormField, FormItem } from '@/components/ui/form'
import { useForm, SubmitHandler } from 'react-hook-form'
import React, { useState } from "react"
import { useTranslation } from "react-i18next"
import { Tabs, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Image, FileText } from "lucide-react"
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectLabel,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select"
import { getDayOfMonthWithSuffix, scheduleTypeOptions } from "@/components/nmap/utils";
import { MultiSelect } from "@/components/ui/multi-select";
import { BaseUser } from "@/components/user/types";
import { dashboard } from "@/components/dashboard/types";
import { useGetUsers } from "@/components/user/queries";
import TimePicker from "@/components/ui/TimePicker";
import { useAddReport, useEditReport } from "@/components/report/queries";
import { useDashboard } from "@/components/dashboard/DashboardContext";
import { useGetDashboards } from "@/components/dashboard/queries";
import { Checkbox } from "@/components/ui/checkbox";

type AddEditReportDialogFormProps = {
  report?: Report,
  users: BaseUser[],
  dashboards?: dashboard[],
  setOpen: (open: boolean) => void,
}

function AddEditReportDialogForm({ report, users, dashboards, setOpen }: AddEditReportDialogFormProps ) {
  
  const { dashboard } = useDashboard();
  const { t } = useTranslation();
  const addReportMutation = useAddReport();
  const editReportMutation = useEditReport();
  const form = useForm<Report>({
    defaultValues: {
      id: report ? report.id : -1,
      dashboard_id: report ? report.dashboard_id : dashboard ? dashboard.id : -1,
      name: report ? report.name : dashboard.id !== -1 ? dashboard.title + ' ' + t('Report'): t('New Report'),
      schedule: report ? report.schedule : 'dailyAt',
      schedule_parameters: (
        report?.schedule_parameters && !Array.isArray(report?.schedule_parameters)
          ? report?.schedule_parameters
          : {}
      ),
      format: report ? report.format : 'pdf',
      recipients: report ? report.users.map((user) => `${user.id}`) : [],
      enabled: report ? report.enabled : true,
    },
  });

  const onSubmit: SubmitHandler<Report> = (data) => {
    if (report) {
      editReportMutation.mutate({ id: data.id, updates: data })
    } else {
      addReportMutation.mutate({ ...data, dashboard_id: dashboard.id === -1 ? Number(data.dashboard_id) : dashboard.id, source: dashboard.id });
    }
    setOpen(false);
  }

  const selectedType = scheduleTypeOptions.find(option => option.value === form.watch('schedule'));
   const dayOptions = [
    { value: '0', label: t('Sunday') },
    { value: '1', label: t('Monday') },
    { value: '2', label: t('Tuesday') },
    { value: '3', label: t('Wednesday') },
    { value: '4', label: t('Thursday') },
    { value: '5', label: t('Friday') },
    { value: '6', label: t('Saturday') },
  ];

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="grid gap-4 py-2">
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel className="text-right">{t('Name')}</FormLabel>
              <FormControl>
                <Input {...field} className="col-span-3 border-border" />
              </FormControl>
            </FormItem>
          )}
        />
        {dashboard.id === -1 && dashboards &&
          <FormField
            control={form.control}
            name="dashboard_id"
            render={({ field }) => (
              <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                <FormLabel>{t('Dashboard')}</FormLabel>
                <FormControl>
                  <Select
                    value={`${field.value === -1 ? '' : field.value}`}
                    onValueChange={field.onChange}
                  >
                    <SelectTrigger className="col-span-3 w-full">
                      <SelectValue placeholder={t('Select dashboard')} />
                    </SelectTrigger>
                    <SelectContent className="max-h-80">
                      {users.map((user) => (
                        <SelectGroup key={user.id}>
                          <SelectLabel>{user.username}</SelectLabel>
                          {dashboards.filter((dashboard) => user.id === dashboard.user_id).map((dashboard) => (
                            <SelectItem key={dashboard.id} value={`${dashboard.id}`}>
                              {dashboard.title}
                            </SelectItem>
                          ))}
                        </SelectGroup>
                      ))}
                    </SelectContent>
                  </Select>
                </FormControl>
              </FormItem>
            )}
          />
        }
        <FormField
          control={form.control}
          name="format"
          render={({ field }) => (
            <Tabs onValueChange={field.onChange} defaultValue={field.value}>
              <FormControl>
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Format')}</FormLabel>
                  <TabsList className="space-x-0">
                    <TabsTrigger className="gap-1 px-2" value={'pdf'}><FileText className="size-4" /> PDF</TabsTrigger>
                    <TabsTrigger className="gap-1 px-2" value={'jpg'}><Image className="size-4" /> JPG</TabsTrigger>
                  </TabsList>
                </FormItem>
              </FormControl>
            </Tabs>
          )}
        />
        <FormField
          control={form.control}
          name="recipients"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t('Recipients')}</FormLabel>
              <FormControl className="col-span-3 w-full">
                <MultiSelect
                  options={users.map((user) => ({ label: `${user.username} (${user.email})`, value: `${user.id}` }))}
                  onValueChange={field.onChange}
                  defaultValue={field.value}
                  placeholder={t("Select recipients")}
                  useSearch={false}
                />
              </FormControl>
            </FormItem>
          )}
        />
        <div className="w-full border-b my-4" />
        <FormField
          control={form.control}
          name="schedule"
          render={({ field }) => (
            <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
              <FormLabel>{t('Schedule')}</FormLabel>
              <FormControl>
                <Select
                  value={field.value}
                  onValueChange={(val) => {
                    field.onChange(val);
                    
                  }}
                >
                  <SelectTrigger className="col-span-3 w-full">
                    <SelectValue placeholder={t('Select schedule type')} />
                  </SelectTrigger>
                  <SelectContent className="max-h-80">
                    <SelectGroup>
                      {scheduleTypeOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectGroup>
                  </SelectContent>
                </Select>
              </FormControl>
            </FormItem>
          )}
        />
        {selectedType?.fields && selectedType.fields.length > 0 && (
          <>
            {selectedType.fields.map(f => {
              if (f.name === 'weekday') {
                return (
                  <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                    <FormLabel>{t(f.label)}</FormLabel>
                    <FormControl>
                      <Select
                        value={form.watch(`schedule_parameters.${f.name}`)}
                        onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                      >
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select weekday')} />
                        </SelectTrigger>
                        <SelectContent className="max-h-80">
                          <SelectGroup>
                            {dayOptions.map(opt => (
                              <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                );
              } else if (f.name === 'month') {
                const monthOptions = [
                  { value: '1', label: t('January') },
                  { value: '2', label: t('February') },
                  { value: '3', label: t('March') },
                  { value: '4', label: t('April') },
                  { value: '5', label: t('May') },
                  { value: '6', label: t('June') },
                  { value: '7', label: t('July') },
                  { value: '8', label: t('August') },
                  { value: '9', label: t('September') },
                  { value: '10', label: t('October') },
                  { value: '11', label: t('November') },
                  { value: '12', label: t('December') },
                ];
                return (
                  <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                    <FormLabel>{t(f.label)}</FormLabel>
                    <FormControl>
                      <Select
                        value={form.watch(`schedule_parameters.${f.name}`) || ''}
                        onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                      >
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select month')} />
                        </SelectTrigger>
                        <SelectContent className="max-h-80">
                          <SelectGroup>
                            {monthOptions.map(opt => (
                              <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                );
              } else if (['Day', 'Day 1', 'Day 2'].includes(f.label)) {
                const dayNumOptions = Array.from({ length: 31 }, (_, i) => ({ value: String(i + 1), label: String(i + 1) }));
                return (
                  <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                    <FormLabel>{t(f.label)}</FormLabel>
                    <FormControl>
                      <Select
                        value={form.watch(`schedule_parameters.${f.name}`) || ''}
                        onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                      >
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select day')} />
                        </SelectTrigger>
                        <SelectContent className="max-h-80">
                          <SelectGroup>
                            {dayNumOptions.map(opt => (
                              <SelectItem key={opt.value} value={opt.value}>{getDayOfMonthWithSuffix(opt.label)}</SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                );
              } else if (['hour1', 'hour2'].includes(f.name)) {
                const hourOptions = Array.from({ length: 24 }, (_, i) => ({ value: String(i), label: String(i).padStart(2, '0') }));
                return (
                  <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                    <FormLabel>{t(f.label)}</FormLabel>
                    <FormControl>
                      <Select
                        value={form.watch(`schedule_parameters.${f.name}`) || ''}
                        onValueChange={val => form.setValue(`schedule_parameters.${f.name}`, val)}
                      >
                        <SelectTrigger className="col-span-3 w-full">
                          <SelectValue placeholder={t('Select hour')} />
                        </SelectTrigger>
                        <SelectContent className="max-h-80">
                          <SelectGroup>
                            {hourOptions.map(opt => (
                              <SelectItem key={opt.value} value={opt.value}>{opt.label}</SelectItem>
                            ))}
                          </SelectGroup>
                        </SelectContent>
                      </Select>
                    </FormControl>
                  </FormItem>
                );
            } else if (f.type === 'time') {
              return (
                <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t(f.label)}</FormLabel>
                  <FormControl className="col-span-3">
                    <TimePicker
                      value={form.watch(`schedule_parameters.${f.name}`) || ''}
                      onChange={(val: string) => form.setValue(`schedule_parameters.${f.name}`, val)}
                      className="w-full"
                    />
                  </FormControl>
                </FormItem>
              );
              } else {
                return (
                  <FormItem key={f.name} className="grid grid-cols-4 items-center gap-4 space-y-0">
                    <FormLabel>{t(f.label)}</FormLabel>
                    <FormControl>
                      <Input
                        type={f.type === 'number' ? 'number' : 'text'}
                        value={form.watch(`schedule_parameters.${f.name}`) || ''}
                        onChange={e => form.setValue(`schedule_parameters.${f.name}`, e.target.value)}
                        className="col-span-3 border-border"
                        placeholder={f.label}
                      />
                    </FormControl>
                  </FormItem>
                );
              }
            })}
          </>
        )} 
        {selectedType?.constraints && (
          <>
            <FormField
              control={form.control}
              name="schedule_parameters.days"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Days')}</FormLabel>
                  <FormControl className="col-span-3 w-full">
                    <MultiSelect
                      options={dayOptions}
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                      placeholder={t("Select days")}
                      maxCount={7}
                      useSearch={false}
                    />
                  </FormControl>
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="schedule_parameters.between"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel>{t('Between')}</FormLabel>
                  <FormControl>
                    <TimePicker 
                      mode="range"
                      value={field.value || ""}
                      onChange={field.onChange}
                    />
                  </FormControl>
                </FormItem>
              )}
            />
          </>
        )}
        {!report &&
          <>
            <div className="w-full border-b my-4" />
            <FormField
              control={form.control}
              name="enabled"
              render={({ field }) => (
                <FormItem className="grid grid-cols-4 items-center gap-4 space-y-0">
                  <FormLabel className="mb-0 cursor-pointer" onClick={() => field.onChange(!field.value)}>
                    {t('Enabled')}
                  </FormLabel>
                  <FormControl>
                    <Checkbox
                      checked={field.value}
                      onCheckedChange={field.onChange}
                    />
                  </FormControl>
                </FormItem>
              )}
            />
          </>
        }
        <div className="w-full border-b my-4" />
        <DialogFooter>
          <Button type="submit" className="justify-end">
            {t('Submit')}
          </Button>
        </DialogFooter>
      </form>
    </Form>
  )
}
export function AddReportDialog({ dashboardId, children }: { dashboardId?: number, children: React.ReactNode }) {
  const [open, setOpen] = useState(false);
  const { t } = useTranslation();
  const { data: users, isLoading: isUsersLoading } = useGetUsers();
  const { data: dashboards } = useGetDashboards();

  if (isUsersLoading || !users) return <div></div>;

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        {children}
      </DialogTrigger>
      <DialogContent className="sm:max-w-[625px]">
        <DialogHeader>
          <DialogTitle>{t('Create Report')}</DialogTitle>
          <DialogDescription>{dashboardId ? t("Use this to create a new scheduled report for this dashboard.") : t('Use this to create a new scheduled report.')}</DialogDescription>
        </DialogHeader>
        <AddEditReportDialogForm setOpen={setOpen} users={users} {...{ dashboardId, dashboards }} />
      </DialogContent>
    </Dialog>
  )
}

export function EditReportDialog({ report, setOpen }: { report: Report, setOpen: (open: boolean) => void }) {
  const { t } = useTranslation();
  const { data: users, isLoading: isUsersLoading } = useGetUsers();
  const { data: dashboards } = useGetDashboards();

  if (isUsersLoading || !users) return <div></div>;

  return (
    <DialogContent className="sm:max-w-[625px]">
      <DialogHeader>
        <DialogTitle>{t('Edit Report')}</DialogTitle>
        <DialogDescription>{t('Use this to edit a scheduled report.')}</DialogDescription>
      </DialogHeader>
      <AddEditReportDialogForm setOpen={setOpen} users={users} {...{ report, dashboards }} />
    </DialogContent>
  )
}