import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { ColumnDef, Row } from "@tanstack/react-table";
import { Ellipsis, Pencil, Trash2 } from "lucide-react";
import { Report } from "@/components/report/types";
import { useTranslation } from "react-i18next";
import { useState } from "react";
import { Link } from "react-router-dom";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { scheduleTypeOptions } from "@/components/nmap/utils";
import { Badge } from "../ui/badge";
import { Switch } from "@/components/ui/switch";
import { useDeleteReport, useEditReport } from "@/components/report/queries";
import { HoverCard, HoverCardTrigger, HoverCardContent } from "@/components/ui/hover-card";
import { EditReportDialog } from "@/components/report/AddEditReportDialog";
import { Dialog, DialogTrigger } from "@/components/ui/dialog";
import { useAuth } from "@/contexts/AuthContext";
import { isAdmin } from "../role/utils";

type ReportsDataTableProps = {
  reports: Report[];
  isLoading?: boolean;
}

export default function ReportsDataTable({ reports, isLoading }: ReportsDataTableProps) {

  const { user } = useAuth();
  const { t } = useTranslation();
  const deleteReportMutation = useDeleteReport();

  const columns: ColumnDef<Report>[] = [
    {
      accessorKey: 'enabled',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Status")} />,
      cell: ({ row }) => {
        const editReportMutation = useEditReport();
        const [toggled, setToggled] = useState(row.original.enabled);
        function handleToggle() {
          setToggled(!toggled);
          editReportMutation.mutate({ id: row.original.id, updates: { enabled: !row.original.enabled } }, {
            onError: () => setToggled(row.original.enabled)
          });
        }
        return (
          <div className="flex items-center ml-2">
            <Switch
              disabled={!isAdmin(user?.role)}
              checked={toggled}
              onCheckedChange={handleToggle}
            />
          </div>
        )
      },
    },
    {
      accessorKey: 'name',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Report Name")} />,
      cell: ({ row }) => (
        <Link className="text-primary" to={`/dashboards/${row.original.dashboard_id}`}>
          {row.original.name}
        </Link>
      ),
    },
    {
      accessorKey: 'schedule',
      header: t('Schedule'),
      cell: ({ row }) => (
        scheduleTypeOptions.find((type) => type.value === row.original.schedule)?.label
      ),
    },
    {
      accessorKey: 'format',
      header: t('Format'),
      cell: ({ row }) => (
        row.original.format.toUpperCase()
      )
    },
    {
      accessorKey: 'recipients',
      header: t('Recipients'),
      cell: ({ row }) => (
        <div className="flex items-center gap-1">
          {
            row.original.users.map((user, index, arr) => {
              if (index < 3 ) return <Badge key={user.id} variant="outline">{`${user.username} (${user.email})`}</Badge>;
              else if (index === 3) return (
                <HoverCard openDelay={50} closeDelay={50} key={user.id}>
                  <HoverCardTrigger>
                    <Badge variant="outline" className="hover:cursor-default">+ {arr.length - 3}</Badge>
                  </HoverCardTrigger>
                  <HoverCardContent side="right" className="p-0 flex flex-col bg-transparent border-none w-fit space-y-1">
                    {arr.slice(3).map((user) => (<Badge key={user.id} variant="outline" className="bg-background">{`${user.username} (${user.email})`}</Badge>))}
                  </HoverCardContent>
                </HoverCard>
              )
            })
          }
        </div>
      )
    },
    ...(isAdmin(user?.role) ? [
      {
        id: 'actions',
        header: '',
        cell: ({ row }: { row: Row<Report> }) => {
          const [open, setOpen] = useState(false);
          return (
            <Dialog open={open} onOpenChange={setOpen}>
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost">
                    <Ellipsis />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DialogTrigger asChild>
                    <DropdownMenuItem>
                      <Pencil />
                      {t('Edit')}
                    </DropdownMenuItem>
                  </DialogTrigger>
                  <DropdownMenuItem onClick={() => deleteReportMutation.mutate(row.original.id)}>
                    <Trash2 />
                    {t('Delete')}
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
              <EditReportDialog report={row.original} setOpen={setOpen} />
            </Dialog>
          )
        },
      },
    ] : [])
  ];

  return (
    <>
      {isLoading ? (
        <DataTableSkeleton></DataTableSkeleton>
      ) : (
        <DataTable columns={columns} data={reports} />
      )}
    </>
  );
}  