import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from "@/components/ui/dropdown-menu";
import { Button } from "@/components/ui/button";
import { DataTable } from "@/components/ui/DataTable/DataTable";
import { ColumnDef, Row } from "@tanstack/react-table";
import { Ellipsis, Pencil, Shield, User, Trash2, Settings } from "lucide-react";
import { Role } from "@/components/role/types";
import { useTranslation } from "react-i18next";
import { useState } from "react";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { useDeleteRole } from "@/components/role/queries";
import { EditRoleDialog } from "@/components/role/AddEditRoleDialog";
import { Dialog, DialogTrigger } from "@/components/ui/dialog";
import { useAuth } from "@/contexts/AuthContext";
import { Badge } from "@/components/ui/badge";
import { HoverCard, HoverCardContent, HoverCardTrigger } from "@/components/ui/hover-card";
import { isAdmin } from "@/components/role/utils";
import { allDefault } from "@/components/role/defaults";

type RolesDataTableProps = {
  roles: Role[];
  isLoading?: boolean;
}

export default function RolesDataTable({ roles, isLoading }: RolesDataTableProps) {

  const { user } = useAuth();
  const { t } = useTranslation();
  const deleteRoleMutation = useDeleteRole();

  const columns: ColumnDef<Role>[] = [
    {
      accessorKey: 'name',
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("Name")} />,
      cell: ({ row }) => (
        <span className="flex items-center gap-2">
          <div className="bg-secondary [&>svg]:size-4 size-8 flex items-center justify-center rounded-full border">
            {row.original.type === 'admin' ? <Shield /> : row.original.type === 'user' ? <User /> : <Settings />}
          </div>
          {row.original.name}
        </span>
      ),
    },
    {
      id: 'user count',
      accessorFn: (row) => row.users.length,
      header: ({ column }) => <DataTableColumnHeader column={column} title={t("User Count")} />,
      cell: ({ row }) => (
        <span>{row.original.users.length}</span>
      ),
    },
    {
      accessorKey: 'users',
      header: t('Users'),
      cell: ({ row }) => (
        <div className="flex items-center gap-1">
          {
            row.original.users.map((user, index, arr) => {
              if (index < 4) return <Badge key={user.id} variant="outline">{`${user.username} (${user.email})`}</Badge>;
              else if (index === 4) return (
                <HoverCard openDelay={50} closeDelay={50} key={user.id}>
                  <HoverCardTrigger>
                    <Badge variant="outline" className="hover:cursor-default">+ {arr.length - 4}</Badge>
                  </HoverCardTrigger>
                  <HoverCardContent side="right" className="p-0 flex flex-col bg-transparent border-none w-fit space-y-1">
                    {arr.slice(4).map((user) => (<Badge key={user.id} variant="outline" className="bg-background">{`${user.username} (${user.email})`}</Badge>))}
                  </HoverCardContent>
                </HoverCard>
              )
            })
          }
        </div>
      )
    },
    {
      accessorKey: 'level',
      header: t("Permission Level"),
      cell: ({ row }) => {
        const role = row.original;

        const enabledPermissions = {
          ...role.flow_source_permissions,
          ...role.suricata_permissions,
          ...role.wireshark_permissions,
          ...role.nmap_permissions,
        };

        const totalPermissions = {
          ...allDefault.flow_source_permissions,
          ...allDefault.suricata_permissions,
          ...allDefault.wireshark_permissions,
          ...allDefault.nmap_permissions,
        }

        let total = 0;
        let enabled = 0;

        Object.entries(totalPermissions).forEach(([key, value]) => {
          const typedKey = key as keyof typeof enabledPermissions;
          const enabledValue = enabledPermissions[typedKey];
          if (Array.isArray(value)) {
            value.forEach((op) => {
              total++;
              if (Array.isArray(enabledValue) && enabledValue.includes(op)) {
                enabled++;
              }
            })
          } else if (typeof value === 'boolean') {
            total++;
            if (enabledPermissions[typedKey] === true) {
              enabled++;
            }
          }
        });

        return (
          <div className="flex items-center gap-2">
            <div className="bg-secondary h-4 w-24 rounded-lg relative">
              <div style={{ width: `${enabled / total * 6}rem`}} className="absolute inset-0 h-4 bg-primary rounded-lg" />
            </div>
            <span className="font-mono">
              {`${Math.floor(enabled / total * 100)}%`}
            </span>
          </div>
        );
      }
    },
    ...(isAdmin(user?.role) ? [
      {
        id: 'actions',
        header: '',
        cell: ({ row }: { row: Row<Role> }) => {
          const [open, setOpen] = useState(false);
          if (row.original.type === 'admin') return <div className="h-8" />;
          return (
            <Dialog open={open} onOpenChange={setOpen}>
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost">
                    <Ellipsis />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DialogTrigger asChild>
                    <DropdownMenuItem>
                      <Pencil />
                      {t('Edit')}
                    </DropdownMenuItem>
                  </DialogTrigger>
                  {!row.original.protected &&
                    <DropdownMenuItem onClick={() => deleteRoleMutation.mutate(row.original.id)}>
                      <Trash2 />
                      {t('Delete')}
                    </DropdownMenuItem>
                  }
                </DropdownMenuContent>
              </DropdownMenu>
              <EditRoleDialog role={row.original} setOpen={setOpen} />
            </Dialog>
          )
        },
      },
    ] : [])
  ];

  return (
    <>
      {isLoading ? (
        <DataTableSkeleton></DataTableSkeleton>
      ) : (
        <DataTable columns={columns} data={roles} />
      )}
    </>
  );
}  