import { useTranslation } from "react-i18next";
import { QueryResponse, QueryResults } from "../types";
import {
    CardContent,
    Card,
} from "@/components/ui/card";
import { ReportSankeyChartWithDialog } from "../../ui/Sankey/SankeyChart";
import { Skeleton } from "../../ui/skeleton"
import { SankeyData, SankeyNode, SankeyLink } from "@/components/ui/Sankey/types.js";

const TOPLINKS = 20;

export function QueryChart({results}: QueryResponse){

  function transformToSankey(queryResults: QueryResults[]):SankeyData {
    const nodeNameToIndex = new Map<string, number>();
    let nodes: SankeyNode[] = [];
    const linksMap = new Map<string, SankeyLink>();

    function getOrAddNode(ip: string, category: "source" | "target" | "other"): number {
      if (!nodeNameToIndex.has(ip+category)) {
        const index = nodes.length;
        nodeNameToIndex.set(ip+category, index);
        nodes.push({ name: ip, category:category, id: ip });
      }
      return nodeNameToIndex.get(ip+category)!;
    }

    for (const result of queryResults) {
      const sourceIndex = getOrAddNode(result.srcip, 'source');
      const targetIndex = getOrAddNode(result.dstip, "target");
      const key = `${sourceIndex}->${targetIndex}`;
      let value = Number(result.bytes);

      if (linksMap.has(key)) {
        linksMap.get(key)!.value += value;
      } else {
        linksMap.set(key, { source: sourceIndex, target: targetIndex, value, sourceId: result.srcip, targetId: result.dstip});
      }
    }

    let links = Array.from(linksMap.values());

    // Removes all but the top links
    const sortedLinks = [...links].sort((a, b) => b.value - a.value);
    links = sortedLinks.slice(0, TOPLINKS);

    const usedNodeIndices = new Set<number>();
    links.forEach(link => {
      usedNodeIndices.add(link.source);
      usedNodeIndices.add(link.target);
    });

    // Keeping unused nodes causes rendering problems
    // They must be removed and the indices in the links
    // must be updated
    const oldToNewIndexMap = new Map<number, number>();
    const filteredNodes: SankeyNode[] = [];
    let newIndex = 0;

    nodes.forEach((node, index) => {
      if (usedNodeIndices.has(index)) {
        oldToNewIndexMap.set(index, newIndex++);
        filteredNodes.push(node);
      }
    });

    const remappedLinks = links
      .filter(link => usedNodeIndices.has(link.source) && usedNodeIndices.has(link.target))
      .map(link => ({
        source: oldToNewIndexMap.get(link.source)!,
        target: oldToNewIndexMap.get(link.target)!,
        value: link.value,
      }));

    nodes= filteredNodes;
    links= remappedLinks;

    return { nodes, links };
  }

  if (!results || results.length === 0) {
      return <div>No data available</div>;
  }
  const data = transformToSankey(results);

  return (
    <ReportSankeyChartWithDialog
      className="h-full flex flex-col"
      sankeyData={data}
      title="srcip → dstip"
      wasPreprocessed={true}
    />
  )
}

export function QueryChartSkeleton(){
  return (
    <>
    <Card>
      <CardContent>
        <Skeleton className="w-full h-80" />
      </CardContent>
    </Card>
    </>
  )
}
