import { useState, useRef } from "react";
import { useTranslation } from "react-i18next";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogClose,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Upload } from "lucide-react";
import { ScanPcapDialogProps } from "@/components/suricata/data/types";
import { toast } from "sonner";

export default function ScanPcapDialog({
  isOpen,
  onOpenChange,
  onSubmit,
}: ScanPcapDialogProps) {
  const { t } = useTranslation();
  const [file, setFile] = useState<File | null>(null);
  const [outputFileName, setOutputFileName] = useState("");
  const inputRef = useRef<HTMLInputElement>(null);

  const sanitizeFileName = (name: string) => {
    const sanitized = name.replace(/\s+/g, "_");
    const extensionIndex = sanitized.lastIndexOf(".");
    return extensionIndex > 0 ? sanitized.substring(0, extensionIndex) : sanitized;
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0] ?? null;
    setFile(selectedFile);
    if (selectedFile) {
      setOutputFileName(sanitizeFileName(selectedFile.name));
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
     if (outputFileName.trim() == "") {
      toast.error(t("Please provide an output filename"));
      return;
    }
    if (!file) {
      toast.error(t("Please select a PCAP file to scan."));
      return;
    }

    onSubmit({
      file,
      outputFileName: outputFileName,
    });
  };

  return (
    <Dialog open={isOpen} onOpenChange={onOpenChange}>
      <DialogContent>
        <form onSubmit={handleSubmit}>
          <DialogHeader>
            <DialogTitle>
              {t(
                "Upload PCAP File to Scan with Suricata"
              )}
            </DialogTitle>
              <DialogDescription className="mb-4">
                {t(
                  "Select a PCAP file to scan with Suricata."
                )}
              </DialogDescription>
            </DialogHeader>
            <div>
              <div className="space-y-4">
                <Label htmlFor="suricata-pcap-upload">
                  {t("PCAP File")}
                </Label>
                <Input
                  type="file"
                  accept=".pcap,.cap,.pcapng"
                  className="hidden"
                  id="suricata-pcap-upload"
                  ref={inputRef}
                  onChange={handleFileChange}
                />
              </div>
              <div className="flex items-center gap-3">
                <Button
                  type="button"
                  onClick={() => inputRef.current?.click()}
                  className="inline-flex items-center gap-2"
                >
                  <Upload className="w-4 h-4" />
                  {t("Upload PCAP")}
                </Button>

                {file && (
                  <div className="text-sm font-bold truncate max-w-[300px]">
                    {file.name}
                  </div>
                )}
              </div>
              <div className="mt-4 space-y-2">
                <Label>{t("Output File Name")}</Label>
                <Input
                  type="text"
                  placeholder={t("Name of new output file")}
                  value={outputFileName}
                  onChange={(e) => setOutputFileName(e.target.value)}
                  onBlur={(e) => setOutputFileName(sanitizeFileName(e.target.value))}
                />
              </div>
            </div>
            <div className="flex justify-end mt-4">
              <Button type="submit" disabled={!file}>
                {t("Scan")}
              </Button>
            </div>
        </form>
      </DialogContent>
    </Dialog>
  );
}
