import { allColumns } from "./dataColumns";
import { DataTable } from "../../ui/DataTable/DataTable";
import DataTableSkeleton from "@/components/ui/DataTable/DataTableSkeleton";
import { useEffect, useState } from "react";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/ui/select";
import { DetailSheet } from "@/components/ui/DetailSheet";
import {
    FormControl,
    FormField,
    FormItem,
    FormLabel,
    Form,
    FormMessage,
} from "@/components/ui/form";
import { logFileSelected, SuricataPcapScanUpdateEvent } from "@/components/suricata/data/types";
import {
    useGetConfig,
    useUpdateConfig,
} from "@/components/suricata/config/queries";
import {
    useGetFullJSONMutation,
    useGetJsonFiles,
    useGetSuricataData,
    useImportPcap,
} from "@/components/suricata/data/queries";
import { useTranslation } from "react-i18next";
import { useForm } from "react-hook-form";
import { SuricataConfigData } from "@/components/suricata/config/types";
import {
  useReactTable,
  getCoreRowModel,
  getSortedRowModel,
  getPaginationRowModel,
  type SortingState,
} from '@tanstack/react-table'
import { SuricataData } from "@/components/suricata/data/types";
import { Button } from "@/components/ui/button";
import { FileScan } from "lucide-react";
import ScanPcapDialog from "@/components/suricata/data/ScanPcapDialog";
import { useQueryClient } from "@tanstack/react-query";
import { useEcho } from "@laravel/echo-react";
import { toast } from "sonner";
import { useAuth } from "@/contexts/AuthContext";
import { ConfigDialog } from "@/components/suricata/data/ConfigDialog";
import { DeleteDataDialog } from "@/components/suricata/data/DeleteDataDialog";
import { hasPermission } from "@/components/role/utils";


export function SuricataDataTable({ initialFilter = "" }: { initialFilter?: string }) {
  const { t } = useTranslation();
  const [sheetOpen, setSheetOpen] = useState(false);
  const [sheetLoading, setSheetLoading] = useState(false);
  const [selectedRowData, setRowData] = useState<Record<
    string,
    unknown
  > | null>(null);
    const { data: jsonFileData, isLoading: isLoadingJsons} = useGetJsonFiles();
  const { data: configData } = useGetConfig();
  const updateConfig = useUpdateConfig();
  const [pageIndex, setPageIndex] = useState(0);
  const [pageSize, setPageSize] = useState(10);
  const [sorting, setSorting]= useState<SortingState>([{id:'', desc: false}]);
  const [filter,setFilter]= useState(initialFilter);
  const [pcapFileOpen, setPcapFileOpen] = useState(false);
  const { mutateAsync: importPcap } = useImportPcap();
  const queryClient = useQueryClient();
  const { data, isLoading } = useGetSuricataData(
    pageSize, 
    pageIndex + 1, 
    sorting[0]?.id || 'time', 
    sorting[0]?.desc ? 'desc' : 'asc',
    filter
  );
  const { user } = useAuth();

  useEffect(() => {
    setPageIndex(0);
  }, [sorting]);

  const logFileForm = useForm<logFileSelected>({
    values: {
      selectedFile: configData?.evelog_name ?? "eve.json",
    }
  })

  useEffect(() => {
  if (configData?.evelog_name) {
    logFileForm.setValue("selectedFile", configData.evelog_name);
    setSelectedFile(configData.evelog_name);
  }
  }, [configData?.evelog_name]);

  const [selectedFile, setSelectedFile] = useState(configData?.evelog_name ?? "")

  // Ensure selectedFile is always something after deleting a file
  useEffect(() => {
    if (jsonFileData && selectedFile) {
      if (!jsonFileData.includes(selectedFile)) {
        const fallback = jsonFileData[0] || ""; 
        setSelectedFile(fallback);

        if (fallback) {
          updateConfig.mutateAsync({
            data: { ...configData!, evelog_name: fallback },
            update: true,
          });
        } else {
          setRowData(null);
          toast.warning("No JSON files available.");
        }
      }
    }
  }, [jsonFileData, selectedFile, updateConfig, configData]);


  const { mutateAsync: fetchFullJSON } = useGetFullJSONMutation();

  const handleFileChange = async (data: SuricataConfigData) => {
    if (jsonFileData?.includes(data.evelog_name)) {
      await updateConfig.mutateAsync({ data, update: true });
    }
  }

  const handleGlobalFilterChange = (value: string) => {
    setPageIndex(0);
    setFilter(value);
  };

  const table = useReactTable<SuricataData['data'][number]>({
      data: data?.data ?? [],
      columns: allColumns,
      state: {
        sorting,
        pagination: { pageIndex, pageSize },
      },
      manualSorting: true,
      manualPagination: true,
      pageCount: data?.total
        ? Math.ceil(data.total / pageSize)
        : 0,
      onSortingChange: (updater) => {
        const next = typeof updater === 'function'
          ? updater(sorting)
          : updater
        setSorting(next)
      },
      onPaginationChange: (updater) => {
        const next = typeof updater === 'function'
          ? updater({ pageIndex, pageSize })
          : updater
        setPageIndex(next.pageIndex)
        setPageSize (next.pageSize)
      },
      getCoreRowModel:     getCoreRowModel(),
      getSortedRowModel:   getSortedRowModel(),
      getPaginationRowModel:getPaginationRowModel(),
    })

  const handleRowClick = async (row: { id: number }) => {
    setSheetOpen(true);
    setSheetLoading(true);
    const id = row["id"];

    try {
      const jsonData = await fetchFullJSON({ id });
      setRowData(jsonData);
    } catch (error) {
      setRowData({"error": `Full log not found ${configData?.evelog_name ?? "your eve log file"} may have rotated.`});
    } finally {
      setSheetLoading(false);
    }
  };

  useEcho(
  `NNA.Users`,
  'SuricataPcapScanUpdate',
  (event: SuricataPcapScanUpdateEvent) => {
    queryClient.invalidateQueries({ queryKey: ["AllSuricataData"] });
    queryClient.invalidateQueries({ queryKey: ["JsonFiles"] });
    queryClient.invalidateQueries({ queryKey: ["suricataConfig"] });
    if (event.status === 'success') {
      toast.success("Suricata Scan Successful")
    }
    else {
      toast.error(event.message)
    }
  }
);


  if (isLoadingJsons)
    return <DataTableSkeleton />;
  
  return (
    <div>
      <div className="flex items-center justify-between mb-2 gap-4">
        <div className="flex gap-4 flex-1">
          <div className="flex flex-row w-75">
            <Form {...logFileForm}>
              <form>
                <FormField
                  control={logFileForm.control}
                  name="selectedFile"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>
                        {t("Select log file:")}
                      </FormLabel>
                        <Select
                          onValueChange={(value) => {
                            setSelectedFile(value);
                            field.onChange(value);
                            configData!.evelog_name=value
                            handleFileChange(configData!)
                          }}
                          value={selectedFile}
                        >
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent className="max-h-100 scroll-y-auto">
                            {[
                              ...jsonFileData ?? []
                            ].map((option) => (
                              <SelectItem
                                key={option}
                                value={option}
                              >
                                {option}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                    </FormItem>
                  )}
                />
              </form>
            </Form>
            <div className="flex flex-row space-x-3 ml-3 self-end">
              {hasPermission(user?.role, 'suricata_permissions', 'data', 'delete') &&
                <DeleteDataDialog jsonFile={selectedFile} />
              }
              {hasPermission(user?.role, 'suricata_permissions', 'config') &&
                <ConfigDialog />
              }
            </div>
          </div>
        </div>
        {hasPermission(user?.role, 'suricata_permissions', 'scan_pcap') &&
          <Button
            onClick={() => {
                setPcapFileOpen(true);
            }}
          >
            <FileScan />
            {t("Scan PCAP")}
          </Button>
        }
      </div>
      <DataTable
          table={table}
          onRowClick={handleRowClick}
          onGlobalFilterChange={handleGlobalFilterChange}
          defaultGlobalFilter={filter}
          isLoading={isLoading}
      />
      {sheetOpen && (
          <DetailSheet
              isOpen={sheetOpen}
              onOpenChange={setSheetOpen}
              JSONdata={selectedRowData ?? {}}
              title="Suricata Log Entry"
              description="Verbose details of the Suricata log."
              loading={sheetLoading}
          />
      )}
       {pcapFileOpen && (
          <ScanPcapDialog
            isOpen={pcapFileOpen}
            onOpenChange={setPcapFileOpen}
            onSubmit={async ({
                file,
                outputFileName,
            }) => {
              if (file) {
                try {
                  setPcapFileOpen(false);
                  await importPcap({
                    file: file,
                    output_file_name: outputFileName,
                  });
                } catch (error) {
                  console.error("Error processing PCAP:", error);
                }
              }
            }}
          />
        )}
    </div>
  )
}
