import { ColumnDef } from "@tanstack/react-table";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
  TooltipProvider,
} from "@/components/ui/tooltip";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { useState, useRef, useEffect } from "react";
import { SuricataData } from "@/components/suricata/data/types";
import { DataActionDropdown } from "@/components/suricata/data/DataActionDropdown";
import TextWithCopyButton from "@/components/ui/TextWithCopyButton";
import { hasPermission } from "@/components/role/utils";
import { useAuth } from "@/contexts/AuthContext";

const createColumn = <T extends SuricataData["data"][number]>(
  accessorKey: keyof T,
  title: string,
  useCopyButton: boolean = false,
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={title}
    />
  ),
  cell: ({ row }) => {
    const value = String(row.getValue(accessorKey as string));
    const textRef = useRef<HTMLDivElement>(null);
    const [isOverflowing, setIsOverflowing] = useState(false);

    useEffect(() => {
      const checkOverflow = () => {
        const el = textRef.current;
        if (el) {
          const isOver = el.scrollWidth > el.clientWidth;
          setIsOverflowing(isOver);
        }
      };

    let resizeTimeout: NodeJS.Timeout;
    const handleResize = () => {
      clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(() => {
        checkOverflow();
      }, 100);
    };

    checkOverflow();
    window.addEventListener("resize", handleResize);

    return () => {
        window.removeEventListener("resize", handleResize);
        clearTimeout(resizeTimeout);
      };
    }, [value]);

    const TextContent = (
      <div
        ref={textRef}
        className="pl-1 truncate whitespace-nowrap overflow-hidden text-ellipsis w-full max-w-[160px]"
      >
        {value}
      </div>
    );

    return isOverflowing ? (
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
          <TooltipContent>
            <div className="flex items-center gap-2">
              <p>{value}</p>
              {useCopyButton ? <TextWithCopyButton text={value} /> : null}
            </div>
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    ) : (
      useCopyButton ? (
        <TextWithCopyButton text={value} />
      ) : TextContent
    );
  },
});

const createActionColumn = <T extends SuricataData["data"][number]>(): ColumnDef<T> => ({
  id: "actions",
  cell: ({ row }) => {
    const data = row.original;
    const src = String(data["source IP" as keyof T]);
    const dest = String(data["destination IP" as keyof T]);
    const { user } = useAuth();

    if (!hasPermission(user?.role, 'suricata_permissions', 'data', 'put')) return;

    return (
      <DataActionDropdown
        src={src}
        dest={dest}
      />
    );
  },
});

export const allColumns: ColumnDef<SuricataData["data"][number]>[] = 
  [
    createColumn("interface", "Interface"),
    createColumn("time", "Time"),
    createColumn("flow ID", "Flow ID"),
    createColumn("event type", "Event Type"),
    createColumn("source IP", "Source IP", true),
    createColumn("source port", "Source Port"),
    createColumn("destination IP", "Destination IP", true),
    createColumn("destination port", "Destination Port"),
    createColumn("protocol", "Protocol"),
    createActionColumn<SuricataData["data"][number]>(),
  ];

