import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogTitle,
  DialogClose,
  DialogHeader,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { useTranslation } from "react-i18next";
import { useState } from "react";
import { usePostScanAction } from "@/components/suricata/queries";
import { Play } from "lucide-react";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { useForm } from "react-hook-form";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
 } from "@/components/ui/select";
import { z } from "zod";



export function StartSuricataDialog({interfaces}: {interfaces: string[] | undefined}) {
  const { t } = useTranslation();
  const { mutate: startStopScan } = usePostScanAction();
  const [error, setError] = useState<string | null>(null);
  const [open, setOpen] = useState(false);
  if (!interfaces) {
    interfaces = ['custom'];
  } else if (!interfaces.includes('custom')) {
    interfaces.push('custom');
  }

  const form = useForm<{ networkInterface: string; customInterface: string }>({
      defaultValues: {
        networkInterface: "",
        customInterface: ""
      }
    });
  
  const handleSubmit = form.handleSubmit((data) => {
    const scheme = z.object({
      networkInterface: z.string().min(1, t("Network interface is required to start a scan.")),
      customInterface: z.string().optional(),
    }).refine(
      (data) => {
        if (data.networkInterface === "custom") {
          return data.customInterface && data.customInterface.trim().length > 0;
        }
        return true; 
      },
      {
        message: t("Custom interface name is required."),
      }
    );
    const result = scheme.safeParse(data);
    if (!result.success) {
      const errorMessages = result.error.issues.map((err) => err.message).join("\n");
      setError(errorMessages);
      return;
    }
    if (data.networkInterface === "custom") {
      data.networkInterface = data.customInterface;
    }
    setOpen(false);
    const networkInterface = data.networkInterface.trim();
    startStopScan({ action: "start", networkInterface });
  });

return (
    <Dialog open={open} onOpenChange={setOpen}>
        <DialogTrigger asChild>
          <Button className="mt-2">
            <Play />
          </Button>
        </DialogTrigger>
      <DialogContent className="flex flex-col">
        <DialogHeader>
          <DialogTitle>{t("Suricata Status")}</DialogTitle>
          <DialogDescription>
            {t("Start your Suricata instance")}
          </DialogDescription>
        </DialogHeader>
        {error && <div className="text-destructive text-sm mb-2">{error}</div>}
        <Form {...form}>
          <form id="start-suricata-form" onSubmit={handleSubmit} className="space-y-4">
            <FormField
              control={form.control}
              name="networkInterface"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t("Network Interface")}</FormLabel>
                  <FormControl>
                    <Select
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                    >
                      <SelectTrigger className="w-full">
                        <SelectValue
                          placeholder={t("Select a network interface")}
                        />
                      </SelectTrigger>
                      <SelectContent>
                        {interfaces.map((iface) => (
                          <SelectItem key={iface} value={iface}>
                            {iface}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            {(form.watch("networkInterface") === "custom") && (
            <FormField
              control={form.control}
              name="customInterface"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>{t("Custom Interface")}</FormLabel>
                  <FormControl>
                    <Input
                      placeholder={t("Enter interface name (e.g., eth0)")}
                      {...field}
                    />
                  </FormControl>
                </FormItem>
              )}
            />
          )}
          </form>
        </Form>
        <div className="flex justify-end gap-4">
          <DialogClose asChild>
            <Button variant="outline">{t("Close")}</Button>
          </DialogClose>
          <Button type="submit" form="start-suricata-form">
             {t("Start Suricata")}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  )
}