import { Card, CardContent } from "@/components/ui/card";
import InfoCard from "@/components/ui/InfoCard";
import { Shield, Clock, CircleAlert, Server } from "lucide-react";
import { useTranslation } from "react-i18next";
import { CategoryChart } from "./CategoryChart";
import { SeverityPieChart } from "./SeverityPieChart";
import { TopAlertsTable } from "./TopAlerts";
import { useGetSuricataDailyAlertsCount } from "../alerts/queries";
import { useGetScanDetails, useGetNetworkInterfaces, useIsRunning } from "@/components/suricata/queries";
import { Skeleton } from "@/components/ui/skeleton";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { hasPermission } from "@/components/role/utils";
import { StartSuricataDialog } from "./StartSuricataDialog";
import { useAuth } from "@/contexts/AuthContext";
import StopSuricataAlertDialog from "@/components/suricata/overview/StopSuricataAlertDialog";

export function SuricataOverview() {
  const { t } = useTranslation();
  const { data: totalAlerts, isLoading: totalAlertsIsLoading } = useGetSuricataDailyAlertsCount();
  const { data: scanDetails, isLoading: scanDetailsIsLoading } = useGetScanDetails();
  const { data: networkInterfaces, isLoading: networkInterfacesLoading } = useGetNetworkInterfaces(); 
  const { user } = useAuth();
  useIsRunning();
  const dataForStatus = (
    <div className="flex items-center gap-2">
      {scanDetails?.running ? (
        <>
          <p>{t("Active")}</p>
          {hasPermission(user?.role, 'suricata_permissions', 'start_stop_scan') &&
            <TooltipProvider>
              <Tooltip>
                <TooltipTrigger asChild>
                  <StopSuricataAlertDialog />
                </TooltipTrigger>
                <TooltipContent>
                  <p>{t("Stop Suricata Scan")}</p>
                </TooltipContent>
              </Tooltip>
            </TooltipProvider>
          }
        </>
      ) : (
        <>
          <p>{t("Inactive")}</p>

          {hasPermission(user?.role, 'suricata_permissions', 'start_stop_scan') &&
            <TooltipProvider>
              <Tooltip>
                <TooltipTrigger asChild>
                  <StartSuricataDialog interfaces={networkInterfaces} />
                </TooltipTrigger>
                <TooltipContent>
                  <p>{t("Start Suricata Scan")}</p>
                </TooltipContent>
              </Tooltip>
            </TooltipProvider>
          }
        </>
      )}
    </div>
  );
  if (scanDetailsIsLoading || totalAlertsIsLoading || networkInterfacesLoading) {
    return (
      <Card className="w-full h-full overflow-x-auto">
        <CardContent className="w-full h-full">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 w-full h-full justify-items-center">
            <Skeleton
              className="rounded-lg"
              style={{
                height: "150px",
                width: "100%",
              }}
            />
            <Skeleton
              className="rounded-lg"
              style={{
                height: "150px",
                width: "100%",
              }}
            />
            <Skeleton
              className="rounded-lg"
              style={{
                height: "150px",
                width: "100%",
              }}
            />
            <Skeleton
              className="rounded-lg"
              style={{
                height: "450px",
                width: "90%",
              }}
            />
            <Skeleton
              className="rounded-lg"
              style={{
                height: "450px",
                width: "100%",
              }}
            />
            <Skeleton
              className="rounded-lg"
              style={{
                height: "450px",
                width: "90%",
              }}
            />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardContent className="flex flex-col gap-4">
        <div className="w-full grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-4">
          <InfoCard
            header={t("Suricata Status")}
            data={dataForStatus}
            icon={<Shield />}
            tooltip={t("Status of the Suricata process.")}
          />
          <InfoCard
            header={t("Scan Start Time")}
            data={scanDetails?.started_at ?? "----/--/-- --:--:--"}
            icon={<Clock />}
            tooltip={t(
              "The time the current Suricata process started."
            )}
          />
          <InfoCard
            header={t("Total Weekly Alerts")}
            data={totalAlerts}
            icon={<CircleAlert />}
            tooltip={t(
              "Total number of alerts scanned in the past day."
            )}
          />
          <InfoCard
            header={t("Interface")}
            data={scanDetails?.interface ?? "None"} 
            icon={<Server />}
            tooltip={t("The network interface Suricata is monitoring.")}
          />
        </div>
        <div className="w-full grid grid-cols-1 xl:grid-cols-3 gap-4">
          <CategoryChart />
          <div className="w-full grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-2 xl:col-span-2 gap-4">
            <SeverityPieChart className="w-full" />
            <TopAlertsTable className="w-full" />
          </div>
        </div>
      </CardContent>
    </Card>
  );
}
