import axios from "@/lib/axios";
import { useQuery,useMutation, useQueryClient } from "@tanstack/react-query";
import {
  SuricataVersion,
} from "@/components/suricata/types";
import { toast } from "sonner";

export function useCheckSuricata() {
  return useQuery<SuricataVersion>({
    queryKey: ["hasSuricata"],
    queryFn: async () => {
      const response = await axios.get<SuricataVersion>(
        "/api/v1/suricata/config/version"
      );
      return response.data;
    },
  });
}

type ScanDetails = {
  running: boolean;
  started_at: string;
  current_eve_log: string;
  interface: string | null;
}

export function useGetScanDetails() {
  return useQuery<ScanDetails>({
    queryKey: ["ScanDetails"],
    queryFn: async () => {
      const response = await axios.get<ScanDetails>(
        "/api/v1/suricata"
      );
      return response.data;
    },
    refetchInterval: 1 * 60 * 1000,
  });
}

export function usePostScanAction() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ action, networkInterface }: { action: "start" | "stop"; networkInterface?: string }) => {
      if (action === "start") {
        const response = await axios.post("/api/v1/suricata", { action, networkInterface });
        return response.data;
      }
      const response = await axios.post("/api/v1/suricata", { action });
      return response.data;
    },
    onSuccess: (_, data,) => {
      const message = data.action === "start"
        ? "Scan started successfully."
        : "Scan stopped successfully.";
      toast.success(message);
      queryClient.invalidateQueries({ queryKey: ["ScanDetails"] });
    },
    onError: (error, data) => {
      const message = data.action === "start"
        ? `Failed to start scan: ${error.message}`
        : `Failed to stop scan: ${error.message}`;
      toast.error(message);
    }
  });
}

export function useIsRunning() {
  const queryClient = useQueryClient();

  return useQuery({
    queryFn: async () => {
      const response = await axios.get("/api/v1/suricata/is-running");
      queryClient.invalidateQueries({ queryKey: ["ScanDetails"] });
      return response.data;
    },
    queryKey: ["IsSuricataRunning"],
  });
}

export function useGetNetworkInterfaces() {
  return useQuery<string[]>({
    queryKey: ["NetworkInterfaces"],
    queryFn: async () => {
      const response = await axios.get<{interfaces: string[]}>(
        "/api/v1/suricata/interfaces"
      );
      return response.data.interfaces;
    },
  });
}