import { ColumnDef } from "@tanstack/react-table";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
  TooltipProvider,
} from "@/components/ui/tooltip";
import { DataTableColumnHeader } from "@/components/ui/DataTable/DataTableColumnHeader";
import { SuricataRuleset } from "@/components/suricata/rules/rulesets/types";
import { useState, useRef, useEffect } from "react";
import { Switch } from "@/components/ui/switch"
import { Badge } from "@/components/ui/badge";
import { usePatchRuleset } from "@/components/suricata/rules/rulesets/queries";
import { RulesetActionsDropdown } from "./RulesetActionsDropdown";
import { Checkbox } from "@/components/ui/checkbox";
import { useAuth } from "@/contexts/AuthContext";
import { hasPermission } from "@/components/role/utils";
import { toast } from "sonner";
import { useTranslation } from "react-i18next";

const createColumn = <T extends SuricataRuleset>(
  accessorKey: keyof T,
  title: string
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={title}
    />
  ),
  cell: ({ row }) => {
    const value = String(row.getValue(accessorKey as string));
    const textRef = useRef<HTMLDivElement>(null);
    const [isOverflowing, setIsOverflowing] = useState(false);

    useEffect(() => {
      const checkOverflow = () => {
        const el = textRef.current;
        if (el) {
          const isOver = el.scrollWidth > el.clientWidth;
          setIsOverflowing(isOver);
        }
      };

        let resizeTimeout: NodeJS.Timeout;
        const handleResize = () => {
          clearTimeout(resizeTimeout);
          resizeTimeout = setTimeout(() => {
            checkOverflow();
          }, 100);
        };

        checkOverflow();
        window.addEventListener("resize", handleResize);

        return () => {
          window.removeEventListener("resize", handleResize);
          clearTimeout(resizeTimeout);
        };
    }, [value]);

    const TextContent = (
      <div
        ref={textRef}
        className="pl-1 truncate whitespace-nowrap overflow-hidden text-ellipsis w-full max-w-[300px]"
      >
        {value}
      </div>
    );

    return isOverflowing ? (
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>{TextContent}</TooltipTrigger>
          <TooltipContent>
            <p>{value}</p>
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    ) : (
      TextContent
    );
  },
});

const createLabelColumn = <T extends SuricataRuleset>(
  accessorKey: keyof T,
  title: string
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={title}
    />
  ),
  cell: ({ row }) => {
    const rulesetType = row.getValue(accessorKey as string)
    return (
    <Badge variant={rulesetType == 'Open' ? "green" : rulesetType == "Commercial" ? "red" : "blue"} className="w-3/4">
      {row.getValue(accessorKey as string)}
    </Badge>
    );
  }
})

const createEnableSwitchColumn = <T extends SuricataRuleset>(
  accessorKey: keyof T,
  title: string
): ColumnDef<T> => ({
  accessorKey,
  header: ({ column }) => (
    <DataTableColumnHeader
      className="text-muted-foreground"
      column={column}
      title={title}
    />
  ),
  cell: ({ row }) => {
    const value = row.getValue(accessorKey as string);
    const { mutate: patchRuleset } = usePatchRuleset();
    const id = row.original.id;
    const { user } = useAuth();
    const { t } = useTranslation();

    return (
      <Switch
        disabled={!hasPermission(user?.role, 'suricata_permissions', 'rulesets', 'put')}
        checked={value as boolean}
        onCheckedChange={(checked) => {
          if (row.original.secretCodeRequired && !row.original.secretCode) {
            toast.error(t("A secret code is required to enable this ruleset. Edit the ruleset to add a secret code."));
          } else {
            patchRuleset({ ids: id, updates: { enabled: Boolean(checked) } });
          }
        }}
      />
    );
  },
});

const actionsButton: ColumnDef<SuricataRuleset> = {
    id: "actions",
    cell: ({ row }) => {
      return (
        <RulesetActionsDropdown
          id={row.original.id}
          name={row.original.name}
        />
      );
    }
  };

const checkBox: ColumnDef<SuricataRuleset> = {
  id: "select",
  header: ({ table }) => (
    <Checkbox
      checked={
        table.getIsAllPageRowsSelected() ||
        (table.getIsSomePageRowsSelected() && "indeterminate")
      }
      onCheckedChange={(value) => table.toggleAllPageRowsSelected(!!value)}
      aria-label="Select all"
    />
  ),
  cell: ({ row }) => (
    <Checkbox
      checked={row.getIsSelected()}
      onCheckedChange={(value) => row.toggleSelected(!!value)}
      aria-label="Select row"
    />
  ),
  enableSorting: false,
  enableHiding: false,
}

export const rulesetColumns: ColumnDef<SuricataRuleset>[] = 
[
  checkBox,
  createLabelColumn("type", "Type"),
  createColumn("name", "Name"),
  createColumn("description", "Description"),
  createColumn("Last Updated", "Last Updated"),
  createColumn("Fetch Status", "Fetch Status"),
  createEnableSwitchColumn("enabled", "Enabled"),
  actionsButton,
]