import { useState, Fragment } from "react";
import { useTranslation } from "react-i18next";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { ChevronDown, ChevronRight } from "lucide-react";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetDescription,
} from "@/components/ui/sheet";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { ScrollArea, ScrollBar } from "@/components/ui/scroll-area";
import { DetailSheetSkeleton } from "@/components/suricata/DetailSheetSkeleton";

interface NestedTableProps {
  data: any;
  level?: number;
}

export const NestedTable: React.FC<NestedTableProps> = ({
  data,
  level = 0,
  }) => {
  const [openKeys, setOpenKeys] = useState<Record<string, boolean>>({});

  if (typeof data !== "object" || data === null) {
    return null;
  }

  const toggleOpen = (key: string) => {
    setOpenKeys((prev) => ({ ...prev, [key]: !prev[key] }));
  };

  return (
    <>
      {Object.entries(data).map(([key, value]) => {
        const isObject = typeof value === "object" && value !== null;
        const uniqueKey = `${level}-${key}`;
        const indentStyle = { paddingLeft: `${level * 16}px` };

        if (isObject) {
          return (
            <Fragment key={uniqueKey}>
              <TableRow
                className="bg-muted/30 hover:bg-muted/40 transition-colors cursor-pointer"
                onClick={() => toggleOpen(uniqueKey)}
              >
                <TableCell
                  style={indentStyle}
                  className="text-sm font-semibold text-primary w-1/2"
                >
                  {openKeys[uniqueKey] ? (
                    <ChevronDown className="inline mr-2" />
                  ) : (
                    <ChevronRight className="inline mr-2" />
                  )}
                  {key}
                </TableCell>
                  <TableCell className="w-1/2 text-sm italic text-muted-foreground">
                    {Array.isArray(value)
                      ? `[${value.length} items]`
                      : "{...}"}
                  </TableCell>
              </TableRow>
                {openKeys[uniqueKey] && (
                  <NestedTable data={value} level={level + 1} />
                )}
            </Fragment>
          );
        }

        return (
          <TableRow key={uniqueKey}>
            <TableCell
              style={indentStyle}
              className="text-sm font-normal w-1/2"
            >
              {key}
            </TableCell>
              <TableCell className="text-sm text-foreground max-w-[300px] p-0 w-1/2">
                {typeof value === "string" && value.length > 80 ? (
                  <ScrollArea className="w-full p-2">
                    <span className="break-all">{value}</span>
                    <ScrollBar
                      orientation="horizontal"
                      className="mt-2"
                    />
                  </ScrollArea>
                  ) : (
                  <span className="p-2 block">
                    {String(value)}
                  </span>
                  )}
              </TableCell>
            </TableRow>
        );
      })}
    </>
  );
};

interface DetailSheetProps {
  isOpen: boolean;
  onOpenChange: (open: boolean) => void;
  JSONdata: Record<string, unknown>;
  title: string;
  description: string;
  loading: boolean;
}

export function DetailSheet({
  isOpen,
  onOpenChange,
  JSONdata,
  title,
  description,
  loading,
}: DetailSheetProps) {
  const { t } = useTranslation();
  return (
    <Sheet open={isOpen} onOpenChange={onOpenChange}>
      <SheetContent className="sm:max-w-[800px] p-0">
        <div className="p-4 h-full">
          <SheetHeader>
            <SheetTitle>{title}</SheetTitle>
            <SheetDescription>{description}</SheetDescription>
          </SheetHeader>
            {loading ? (
              <DetailSheetSkeleton
                isOpen={isOpen}
                onOpenChange={onOpenChange}
                title={title}
                description={description}
              />
            ) : (
              <div className="mt-4 h-full">
                <Tabs defaultValue="details" className="h-full">
                  <div className="flex items-center space-x-2 mb-2">
                    <TabsList className="w-full">
                      <TabsTrigger
                        value="details"
                        className="w-1/2"
                      >
                        {t("Details")}
                      </TabsTrigger>
                      <TabsTrigger
                        value="json"
                        className="w-1/2"
                      >
                        {t("JSON")}
                      </TabsTrigger>
                    </TabsList>
                  </div>
                  <ScrollArea className="h-[calc(100%-150px)]">
                    <TabsContent value="details">
                      <Table className="w-full">
                        <TableHeader>
                          <TableRow>
                            <TableHead>
                              {t("Field")}
                            </TableHead>
                            <TableHead>
                              {t("Value")}
                            </TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          <NestedTable data={JSONdata} />
                        </TableBody>
                      </Table>
                    </TabsContent>

                    <TabsContent value="json">
                      <pre className="whitespace-pre-wrap break-all p-4">
                        {JSON.stringify(JSONdata, null, 2)}
                      </pre>
                    </TabsContent>
                  </ScrollArea>
                </Tabs>
              </div>
            )}
        </div>
      </SheetContent>
    </Sheet>
  );
}
