import * as React from "react";
import { useTranslation } from "react-i18next";
import { cva, type VariantProps } from "class-variance-authority";
import { ChevronsUpDown, Check } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import { ScrollArea } from "@/components/ui/scroll-area";

/**
 * Variants for the combobox component to handle different styles.
 * Uses class-variance-authority (cva) to define different styles based on "variant" prop.
 */
const comboboxVariants = cva(
  "",
  {
    variants: {
      variant: {
        default: 
          "border-foreground/10 text-foreground bg-card hover:bg-card/80",
        secondary:
          "border-foreground/10 bg-secondary text-secondary-foreground hover:bg-secondary/80",
        destructive:
          "border-transparent bg-destructive text-destructive-foreground hover:bg-destructive/80",
        inverted: "inverted",
      },
    },
    defaultVariants: {
      variant: "default",
    },
  }
);

/**
 * Props for Combobox component
 */
export interface ComboboxOption {
  label: string;
  value: string;
  icon?: React.ComponentType<{ className?: string }>;
}

interface ComboboxProps
  extends Omit<React.ButtonHTMLAttributes<HTMLButtonElement>, "onChange">,
    VariantProps<typeof comboboxVariants> {
  /**
   * Array of options to display in the combobox.
   */
  options: ComboboxOption[];
  /**
   * The selected value (for controlled usage).
   */
  value?: string;
  /**
   * Callback when the value changes.
   */
  onChange?: (value: string) => void;
  /**
   * The default value (for uncontrolled usage).
   */
  defaultValue?: string;
  /**
   * Placeholder text when no value is selected.
   */
  placeholder?: string;
  /**
   * If true, renders the combobox as a child of another component.
   */
  asChild?: boolean;
  /**
   * Additional class names for custom styles.
   */
  className?: string;
  /**
   * If true, disables the combobox.
   */
  disabled?: boolean;
  /**
   * 
   */
  showNoneOption?: boolean;
}

export const Combobox = React.forwardRef<HTMLButtonElement, ComboboxProps>(
  (
    {
      options,
      value: controlledValue,
      onChange,
      defaultValue = "",
      placeholder = "Select option",
      variant,
      asChild = false,
      className,
      disabled = false,
      showNoneOption = false,
      ...props
    },
    ref
  ) => {
    const [uncontrolledValue, setUncontrolledValue] = React.useState<string>(defaultValue);
    const [isPopoverOpen, setIsPopoverOpen] = React.useState(false);
    const { t } = useTranslation();
    const value = controlledValue !== undefined ? controlledValue : uncontrolledValue;

    // Ref for the trigger button
    const triggerRef = React.useRef<HTMLButtonElement>(null);
    // State for popover width
    const [popoverWidth, setPopoverWidth] = React.useState<number | undefined>(undefined);

    React.useEffect(() => {
      if (isPopoverOpen && triggerRef.current) {
        setPopoverWidth(triggerRef.current.offsetWidth);
      }
    }, [isPopoverOpen]);

    const handleSelect = (val: string) => {
      if (onChange) {
        onChange(val);
      }
      if (controlledValue === undefined) {
        setUncontrolledValue(val);
      }
      setIsPopoverOpen(false);
    };

    const selectedOption = options.find((o) => o.value === value);

    return (
      <Popover open={isPopoverOpen} onOpenChange={setIsPopoverOpen}>
        <PopoverTrigger asChild>
          <Button
            ref={(node) => {
              if (typeof ref === "function") ref(node);
              triggerRef.current = node;
            }}
            type="button"
            role="combobox"
            aria-expanded={isPopoverOpen}
            aria-controls="combobox-listbox"
            disabled={disabled}
            className={cn(
              "flex w-full p-1 rounded-md border min-h-10 h-auto items-center justify-between bg-inherit hover:bg-inherit [&_svg]:pointer-events-auto",
              className,
            )}
            {...props}
          >
            {selectedOption ? (
              <div className="flex items-center w-full">
                {selectedOption.icon && (
                  <selectedOption.icon className="h-4 w-4 mr-2" />
                )}
                <span className="text-foreground">{selectedOption.label}</span>
              </div>
            ) : (
              <span className="text-sm text-muted-foreground">{placeholder}</span>
            )}
            <ChevronsUpDown className="h-4 mx-2 cursor-pointer text-muted-foreground" />
          </Button>
        </PopoverTrigger>
        <PopoverContent
          className={cn(
            "p-0 min-w-[12rem] w-full overflow-visible",
            comboboxVariants({ variant })
          )}
          align="start"
          id="combobox-listbox"
          style={popoverWidth ? { width: popoverWidth } : undefined}
        >
          <Command>
            <CommandInput placeholder={t("Search...")} />
            <ScrollArea className="h-72 rounded-md overflow-x-hidden">
              <CommandList className="pointer-events-auto !overflow-y-visible !max-h-none">
                <CommandEmpty>No options found.</CommandEmpty>
                <CommandGroup>
                  {showNoneOption && (
                    <CommandItem
                      key="none"
                      value="None"
                      keywords={["None"]}
                      onSelect={() => handleSelect("")}
                      className="cursor-pointer flex items-center justify-between"
                    >
                      <div className="flex items-center">
                        <span className="italic text-muted-foreground">None</span>
                      </div>
                      {value === "" && (
                        <Check className="h-4 w-4 text-primary ml-2" />
                      )}
                    </CommandItem>
                  )}
                  {options.map((option) => (
                    <CommandItem
                      key={option.value}
                      value={option.value}
                      keywords={[option.label]}
                      onSelect={() => handleSelect(option.value)}
                      className="cursor-pointer flex items-center justify-between"
                    >
                      <div className="flex items-center">
                        {option.icon && (
                          <option.icon className="mr-2 h-4 w-4 text-muted-foreground" />
                        )}
                        <span>{option.label}</span>
                      </div>
                      {value === option.value && (
                        <Check className="h-4 w-4 text-primary ml-2" />
                      )}
                    </CommandItem>
                  ))}
                </CommandGroup>
              </CommandList>
            </ScrollArea>
          </Command>
        </PopoverContent>
      </Popover>
    );
  }
);

Combobox.displayName = "Combobox";