import React, { useState, useEffect, useRef } from "react";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectGroup,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Card,
  CardContent,
} from "@/components/ui/card";
import { RadioGroup } from '@radix-ui/react-radio-group';
import { RadioGroupItem } from '@/components/ui/radio-group';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from '@/components/ui/label';
import { PasswordInput } from '@/components/ui/passwordInput';
import ThemesDropdown from "@/components/ui/ThemesDropdown";
import { useForm, SubmitHandler } from "react-hook-form";
import { useTranslation } from "react-i18next";
import { useCreateUser, useEditUser } from "@/components/user/queries";
import { useGetAuthServers } from "@/components/administration/queries";
import type { Language, User, UserFormData } from "@/components/user/types";
import type { AuthServer } from "@/components/administration/types";
import { cn } from "@/lib/utils";
import {
  CircleUser,
  Check,
  Contact,
  Paintbrush,
  Shield,
  KeyRound,
  ChevronsUpDown,
  User as UserIcon,
  Settings,
  Plus,
} from "lucide-react"
import { PageSeparator } from "@/components/Page";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarProvider,
} from "@/components/ui/sidebar";
import {
  Breadcrumb,
  BreadcrumbItem,
  BreadcrumbList,
  BreadcrumbPage,
  BreadcrumbSeparator,
} from "@/components/ui/breadcrumb"
import { Link } from "react-router-dom";
import { useGetRoles } from "@/components/role/queries";
import { ScrollingContainer } from "@/components/ui/scrolling-container";
import { Avatar, AvatarFallback, AvatarImage } from "../ui/avatar";

export default function AddEditUserDialog({ 
    user, 
    children,
    open,
    onOpenChange
}: { 
    user?: User;
    children?: React.ReactNode;
    open?: boolean;
    onOpenChange?: (open: boolean) => void;
}) {
    const [internalOpen, setInternalOpen] = useState(false);
    const [languageOpen, setLanguageOpen] = useState(false);
    const [authType, setAuthType] = useState<'Local' | 'LDAP' | 'AD'>(user?.type ?? 'Local');
    const { t } = useTranslation();
    const createUser = useCreateUser();
    const updateUser = useEditUser();
    const { data: authServers } = useGetAuthServers();
    const sectionRefs = useRef<Record<string, HTMLDivElement | null>>({});
    const [activeSection, setActiveSection] = useState("account-info");
    const containerRef = useRef<HTMLDivElement | null>(null);
    const { data: roles } = useGetRoles();

    // Use external state if provided, otherwise use internal state
    const isOpen = open !== undefined ? open : internalOpen;
    const handleOpenChange = onOpenChange || setInternalOpen;

    const ldapServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ldap");
    const adServers = (authServers ?? []).filter((server: AuthServer) => server.type === "ad");

    const languages: { label: string; value: Language }[] = [
        { label: t("Bulgarian"), value: "bg_BG" },
        { label: t("Czech"), value: "cs_CZ" },
        { label: t("German"), value: "de_DE" },
        { label: t("English"), value: "en_US" },
        { label: t("Spanish"), value: "es_ES" },
        { label: t("French"), value: "fr_FR" },
        { label: t("Italian"), value: "it_IT" },
        { label: t("Japanese"), value: "ja_JP" },
        { label: t("Korean"), value: "ko_KR" },
        { label: t("Polish"), value: "pl_PL" },
        { label: t("Portugese"), value: "pt_PT" },
        { label: t("Russian"), value: "ru_RU" },
        { label: t("Chinese (China)"), value: "zh_CN" },
        { label: t("Chinese (Taiwan)"), value: "zh_TW" },
    ];

    const nav = [
        { name: t("Account Information"), icon: CircleUser, fragment: "account-info" },
        { name: t("Display"), icon: Paintbrush, fragment: "display" },
        { name: t("Details"), icon: Contact, fragment: "details" },
        { name: t("Authentication"), icon: KeyRound, fragment: "authentication" },
        { name: t("Security"), icon: Shield, fragment: "security" },
    ] as const;

    type NavFragment = typeof nav[number]['fragment'];

    const sectionFieldMap: Record<NavFragment, (keyof UserFormData)[]> = {
        "account-info": ["username", "email", "password", "confirm_password"],
        "display": ["lang", "theme"],
        "details": ["first_name", "last_name", "company", "phone"],
        "authentication": ["type", "auth_server_id", "auth_server_data"],
        "security": ["role_id", "apiaccess"]
    };

    const sectionHasErrors = (section_name: NavFragment): boolean => {
        const fields = sectionFieldMap[section_name];
        return fields.some(field => form.formState.errors[field]);
    };

    const userSchema = z.object({
        type: z.enum(['Local', 'LDAP', 'AD']),
        first_name: z.string().optional(),
        last_name: z.string().optional(),
        company: z.string().optional(),
        phone: z.e164({ error: t("Invalid phone number") }).optional().or(z.literal("")),
        username: z.string()
            .min(1, t("Username is required"))
            .max(255, t("Username must be less than 255 characters")),
        password: z.string()
            .min(8, t("Password must be at least 8 characters"))
            .optional()
            .or(z.literal("")),
        confirm_password: z.string().optional().or(z.literal("")),
        email: z.email({ error: t("Invalid email address") })
            .max(255, t("Email must be less than 255 characters")),
        lang: z.enum(["bg_BG", "cs_CZ", "de_DE", "en_US", "es_ES", "fr_FR", "it_IT", "ja_JP", "ko_KR", "pl_PL", "pt_PT", "ru_RU", "zh_CN", "zh_TW", "default"]),
        theme: z.string(),
        apiaccess: z.string(),
        auth_server_id: z.number().optional(),
        auth_server_data: z.string().optional(),
        role_id: z.number(),
    }).superRefine((data, ctx) => {
        if (!user && data.type === 'Local' && data.password && data.password !== data.confirm_password) {
            ctx.addIssue({
                code: "custom",
                message: t("Passwords do not match"),
                path: ["confirm_password"],
            });
        }

        if (!user && data.type === 'Local' && !data.password) {
            ctx.addIssue({
                code: "custom",
                message: t("Password is required for new local users"),
                path: ["password"],
            });
        }

        if (data.auth_server_id && !data.auth_server_data) {
            ctx.addIssue({
                code: "custom",
                message: data.type === "LDAP" ? t("User's Full DN is required") : t("AD Username is required"),
                path: ["auth_server_data"],
            });
        }
    }) satisfies z.ZodType<UserFormData>;

    const form = useForm<UserFormData>({
        resolver: zodResolver(userSchema),
        mode: "onChange",
        defaultValues: {
            type: user?.type ?? "Local",
            first_name: user?.first_name ?? "",
            last_name: user?.last_name ?? "",
            company: user?.company ?? "",
            phone: user?.phone ?? "",
            username: user?.username ?? "",
            password: "",
            confirm_password: "",
            email: user?.email ?? "",
            lang: user?.lang ?? "en_US",
            theme: user?.theme ?? "default",
            apiaccess: user?.apiaccess ? "1" : "0",
            auth_server_id: user?.auth_server_id ?? 0,
            auth_server_data: user?.auth_server_data ?? "",
            role_id: user?.role_id ?? 2,
        }
    });

    useEffect(() => {
      if (!authServers) return;

      // Only set default auth server for new users or when user doesn't have an existing auth_server_id
      if (!user || !user.auth_server_id) {
          if (authType == "AD" && adServers.length > 0) {
              form.setValue("auth_server_id", adServers[0].id, { shouldDirty: true });
          } else if (authType == "LDAP" && ldapServers.length > 0) {
              form.setValue("auth_server_id", ldapServers[0].id, { shouldDirty: true });
          }
      }
    }, [authServers, authType, adServers, ldapServers, user]);

    const onSubmit: SubmitHandler<UserFormData> = (data) => {
      if (user) {
          updateUser.mutate({ data: data, id: user.id }, {
              onSuccess: () => {
                handleDialogChange(false);
              }
          });
      } else {
          createUser.mutate({ ...data }, {
              onSuccess: () => {
                handleDialogChange(false);
              }
          });
      }
    };

    const handleDialogChange = (isOpen: boolean) => {
        handleOpenChange(isOpen);
        if (!isOpen) {
            form.reset();
            setAuthType('Local');
        }
    };

    if (!roles) return (
        children
    );

    return (
        <Dialog open={isOpen} onOpenChange={handleDialogChange}>
            {children && (
                <DialogTrigger asChild>
                    {children}
                </DialogTrigger>
            )}
            <DialogContent className="overflow-hidden p-0 sm:max-h-[700px] sm:max-w-[700px] lg:max-w-[1100px]">
                <DialogTitle className="sr-only">
                    {user ? t('Edit User') : t('Create User')}
                </DialogTitle>
                <DialogDescription className="sr-only">
                    {user ? t('Update user information below.') : t('Please enter all fields below to create a new user.')}
                </DialogDescription>
                <SidebarProvider className="items-start">
                    <Sidebar collapsible="none" className="hidden lg:flex">
                        <SidebarContent className="border-r">
                            <SidebarGroup>
                                <SidebarGroupContent>
                                    <SidebarMenu>
                                        {nav.map((item) => (
                                            <SidebarMenuItem key={item.name}>
                                                <SidebarMenuButton
                                                    asChild
                                                    isActive={activeSection === item.fragment}
                                                    className="hover:cursor-pointer"
                                                >
                                                    <a
                                                        onClick={() => {
                                                            setActiveSection(item.fragment);
                                                            const target = sectionRefs.current[item.fragment];
                                                            const container = containerRef?.current?.querySelector(
                                                                '[data-radix-scroll-area-viewport]'
                                                            );
                                                            if (target && container) {
                                                                const containerTop = container.getBoundingClientRect().top;
                                                                const childTop = target.getBoundingClientRect().top;
                                                                const offset = childTop - containerTop;

                                                                container.scrollTo({
                                                                    top: offset + container.scrollTop
                                                                });
                                                            }
                                                        }}
                                                    >
                                                        <item.icon className={sectionHasErrors(item.fragment) ? "text-destructive" : ""} />
                                                        <span className={sectionHasErrors(item.fragment) ? "text-destructive" : ""}>
                                                            {item.name}
                                                        </span>
                                                    </a>
                                                </SidebarMenuButton>
                                            </SidebarMenuItem>
                                        ))}
                                    </SidebarMenu>
                                </SidebarGroupContent>
                            </SidebarGroup>
                        </SidebarContent>
                    </Sidebar>
                    <div className="flex flex-1 flex-col overflow-hidden">
                        <Form {...form}>
                            <form onSubmit={form.handleSubmit(onSubmit)}>
                                <div className="h-16 flex items-center gap-2 px-4">
                                    <Breadcrumb>
                                        <BreadcrumbList>
                                            <BreadcrumbItem className="md:flex gap-4">
                                                <Avatar>
                                                    {user ? (
                                                            <>
                                                                <AvatarImage src={`/storage/avatars/${user?.username}.svg`} className="scale-95" />
                                                                <AvatarFallback>
                                                                    {user?.username.slice(0, 2).toUpperCase()}
                                                                </AvatarFallback>
                                                            </>
                                                        ) : (
                                                            <div className="size-full bg-secondary flex items-center justify-center">
                                                                <Plus className="size-4" />
                                                            </div>
                                                        )
                                                    }
                                                </Avatar>
                                                {t("User")}
                                            </BreadcrumbItem>
                                            <BreadcrumbSeparator className="md:block" />
                                            <BreadcrumbItem>
                                                <BreadcrumbPage>{user ? t('Edit') : t('Create')}</BreadcrumbPage>
                                            </BreadcrumbItem>
                                        </BreadcrumbList>
                                    </Breadcrumb>
                                </div>
                                <ScrollingContainer
                                    setActiveSection={setActiveSection}
                                    threshold={0.5}
                                    sectionRefs={sectionRefs}
                                    containerRef={containerRef}
                                    className="pb-65"
                                >
                                    <Card
                                        ref={(el) => { sectionRefs.current["account-info"] = el }}
                                        id="account-info"
                                    >
                                        <CardContent>
                                            <div className="flex items-center gap-2 text-lg font-semibold mb-6"><CircleUser />{t("Account Information")}</div>
                                            <FormField
                                                control={form.control}
                                                name="username"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('Username')}</FormLabel>
                                                            <FormDescription>{t("Your unique identifier in Network Analyzer")}</FormDescription>
                                                        </div>
                                                        <div className="w-md">
                                                            <FormControl>
                                                                <Input {...field} placeholder='nagiosadmin' disabled={!!user} />
                                                            </FormControl>
                                                            <FormMessage />
                                                        </div>
                                                    </FormItem>
                                                )}
                                            />
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="email"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('Email')}</FormLabel>
                                                            <FormDescription>{t('Your email address for account access')}</FormDescription>
                                                        </div>
                                                        <div className="w-md">
                                                            <FormControl>
                                                                <Input {...field} placeholder="example@domain.com" />
                                                            </FormControl>
                                                            <FormMessage />
                                                        </div>
                                                    </FormItem>
                                                )}
                                            />
                                            {(authType === "Local" && !user) && (
                                                <>
                                                    <PageSeparator />
                                                    <FormField
                                                        control={form.control}
                                                        name="password"
                                                        render={({ field }) => (
                                                            <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                                <div className="space-y-1">
                                                                    <FormLabel>{t('Password')}</FormLabel>
                                                                    <FormDescription>{t('Your secure password for authentication')}</FormDescription>
                                                                </div>
                                                                <div className="w-md">
                                                                    <FormControl>
                                                                        <PasswordInput {...field} />
                                                                    </FormControl>
                                                                    <FormMessage />
                                                                </div>
                                                            </FormItem>
                                                        )}
                                                    />
                                                    <PageSeparator />
                                                    <FormField
                                                        control={form.control}
                                                        name="confirm_password"
                                                        render={({ field }) => (
                                                            <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                                <div className="space-y-1">
                                                                    <FormLabel>{t('Confirm Password')}</FormLabel>
                                                                    <FormDescription>{t('Re-enter your password')}</FormDescription>
                                                                </div>
                                                                <div className="w-md">
                                                                    <FormControl>
                                                                        <PasswordInput {...field} />
                                                                    </FormControl>
                                                                    <FormMessage />
                                                                </div>
                                                            </FormItem>
                                                        )}
                                                    />
                                                </>
                                            )}
                                        </CardContent>
                                    </Card>
                                    <Card
                                        ref={(el) => { sectionRefs.current["display"] = el }}
                                        id="display"
                                    >
                                        <CardContent>
                                            <div className="text-lg font-semibold mb-6 flex items-center gap-2">{<Paintbrush />}{t("Display")}</div>
                                            <FormField
                                                control={form.control}
                                                name="lang"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t("Language")}</FormLabel>
                                                            <FormDescription>
                                                                {t("Set the display language")}
                                                            </FormDescription>
                                                        </div>
                                                        <Popover open={languageOpen} onOpenChange={setLanguageOpen}>
                                                            <PopoverTrigger asChild>
                                                                <FormControl>
                                                                    <Button
                                                                        variant="outline"
                                                                        role="combobox"
                                                                        className={cn(
                                                                            "w-50 justify-between",
                                                                            !field.value && "text-muted-foreground"
                                                                        )}
                                                                    >
                                                                        {field.value
                                                                            ? languages.find(
                                                                                (lang) => lang.value === field.value
                                                                            )?.label
                                                                            : "Select language"}
                                                                        <ChevronsUpDown className="opacity-50" />
                                                                    </Button>
                                                                </FormControl>
                                                            </PopoverTrigger>
                                                            <PopoverContent className="w-50 p-0">
                                                                <Command>
                                                                    <CommandInput placeholder="Search language..." />
                                                                    <CommandList>
                                                                        <CommandEmpty>
                                                                            {t("No language found")}
                                                                        </CommandEmpty>
                                                                        <CommandGroup>
                                                                            {languages.map((language) => (
                                                                                <CommandItem
                                                                                    value={language.label}
                                                                                    key={language.value}
                                                                                    onSelect={() => {
                                                                                        setLanguageOpen(false);
                                                                                        form.setValue("lang", language.value);
                                                                                    }}
                                                                                >
                                                                                    {language.label}
                                                                                    <Check
                                                                                        className={cn(
                                                                                            "ml-auto",
                                                                                            language.value === field.value
                                                                                                ? "opacity-100"
                                                                                                : "opacity-0"
                                                                                        )}
                                                                                    />
                                                                                </CommandItem>
                                                                            ))}
                                                                        </CommandGroup>
                                                                    </CommandList>
                                                                </Command>
                                                            </PopoverContent>
                                                        </Popover>
                                                        <FormMessage />
                                                    </FormItem>
                                                )}
                                            />
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="theme"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('User Theme')}</FormLabel>
                                                            <FormDescription>{t('Set the display theme')}</FormDescription>
                                                        </div>
                                                        <FormControl>
                                                            <div className="w-50">
                                                                <ThemesDropdown field={field} />
                                                            </div>
                                                        </FormControl>
                                                    </FormItem>
                                                )}
                                            />
                                        </CardContent>
                                    </Card>
                                    <Card
                                        ref={(el) => { sectionRefs.current["details"] = el }}
                                        id="details"
                                    >
                                        <CardContent>
                                            <div className="text-lg font-semibold mb-6 flex items-center gap-2"><Contact />{t("Details")}</div>
                                            <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                <div className="space-y-1">
                                                    <FormLabel>{t("Full Name")}</FormLabel>
                                                    <FormDescription>
                                                        {t("Name displayed in interface")}
                                                    </FormDescription>
                                                </div>
                                                <div className="flex gap-2 w-md">
                                                    <FormField
                                                        control={form.control}
                                                        name="first_name"
                                                        render={({ field }) => (
                                                            <FormItem className="w-full">
                                                                <FormControl>
                                                                    <Input placeholder={t("First Name")} {...field} />
                                                                </FormControl>
                                                                <FormMessage />
                                                            </FormItem>
                                                        )}
                                                    />
                                                    <FormField
                                                        control={form.control}
                                                        name="last_name"
                                                        render={({ field }) => (
                                                            <FormItem className="w-full">
                                                                <FormControl>
                                                                    <Input placeholder={t("Last Name")} {...field} />
                                                                </FormControl>
                                                                <FormMessage />
                                                            </FormItem>
                                                        )}
                                                    />
                                                </div>
                                            </FormItem>
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="company"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('Company')}</FormLabel>
                                                            <FormDescription>{t('Name displayed in Nagios Applications')}</FormDescription>
                                                        </div>
                                                        <FormControl className="w-md">
                                                            <Input {...field} placeholder={t('Nagios Enterprises')} />
                                                        </FormControl>
                                                        <FormMessage />
                                                    </FormItem>
                                                )}
                                            />
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="phone"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('Phone Number')}</FormLabel>
                                                            <FormDescription>{t('Contact phone number')}</FormDescription>
                                                        </div>
                                                        <div className="w-md">
                                                            <FormControl>
                                                                <Input {...field} placeholder="+11234567890" />
                                                            </FormControl>
                                                            <FormMessage />
                                                        </div>
                                                    </FormItem>
                                                )}
                                            />
                                        </CardContent>
                                    </Card>
                                    <Card
                                        ref={(el) => { sectionRefs.current["authentication"] = el }}
                                        id="authentication"
                                    >
                                        <CardContent>
                                            <div className="text-lg font-semibold mb-6 flex items-center gap-2">{<KeyRound />}{t("Authentication")}</div>
                                            <p className="text-sm text-muted-foreground mb-4">
                                                {t('User accounts can be authenticated in many different ways either from your local database or external programs such as Active Directory or LDAP. You can set up external authentication servers in the LDAP/AD Integration settings.')}
                                            </p>
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="type"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t("Authentication Type")}</FormLabel>
                                                            <FormDescription>
                                                                {t("Choose how this user will authenticate.")}
                                                            </FormDescription>
                                                        </div>
                                                        <FormControl>
                                                            <Select
                                                                value={field.value}
                                                                onValueChange={(selectedVal) => {
                                                                setAuthType(selectedVal as 'Local' | 'LDAP' | 'AD');
                                                                form.clearErrors("auth_server_data");
                                                                field.onChange(selectedVal);
                                                                
                                                                // Set auth server immediately when auth type changes
                                                                if (selectedVal == "AD" && adServers.length > 0) {
                                                                    form.setValue("auth_server_id", adServers[0].id, { shouldDirty: true });
                                                                } else if (selectedVal == "LDAP" && ldapServers.length > 0) {
                                                                    form.setValue("auth_server_id", ldapServers[0].id, { shouldDirty: true });
                                                                } else {
                                                                    form.setValue("auth_server_id", 0, { shouldDirty: true });
                                                                }
                                                            }}
                                                                disabled={!!user}
                                                            >
                                                                <SelectTrigger className="w-50">
                                                                    <SelectValue placeholder={t("Select authentication type")} />
                                                                </SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectGroup>
                                                                        <SelectItem value="Local">{t("Local")}</SelectItem>
                                                                        <SelectItem value="AD" disabled={adServers.length === 0}>{t("Active Directory")}</SelectItem>
                                                                        <SelectItem value="LDAP" disabled={ldapServers.length === 0}>{t("LDAP")}</SelectItem>
                                                                    </SelectGroup>
                                                                </SelectContent>
                                                            </Select>
                                                        </FormControl>
                                                        <FormMessage />
                                                    </FormItem>
                                                )}
                                            />
                                            {/* Auth Server Selection */}
                                            {(authType === "LDAP" || authType === "AD") && (
                                                <>
                                                    <PageSeparator />
                                                    <FormField
                                                        control={form.control}
                                                        name="auth_server_id"
                                                        render={({ field }) => (
                                                            <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                                <FormLabel>
                                                                    {authType === "LDAP" ? t('LDAP Server') : t('AD Server')}
                                                                </FormLabel>
                                                                <FormControl>
                                                                    <Select
                                                                        value={field.value?.toString()}
                                                                        onValueChange={(selectedVal) => {
                                                                            field.onChange(Number(selectedVal));
                                                                        }}
                                                                    >
                                                                        <SelectTrigger className="w-md">
                                                                            <SelectValue placeholder="Select a server" />
                                                                        </SelectTrigger>
                                                                        <SelectContent>
                                                                            <SelectGroup className="w-full max-h-80">
                                                                                {(authType === "LDAP" ? ldapServers : adServers).map((server: AuthServer) => (
                                                                                    <SelectItem key={server.id} value={server.id.toString()}>
                                                                                        {server.name} ({authType === "LDAP" ? server.host : server.controllers})
                                                                                    </SelectItem>
                                                                                ))}
                                                                            </SelectGroup>
                                                                        </SelectContent>
                                                                    </Select>
                                                                </FormControl>
                                                                <FormMessage />
                                                            </FormItem>
                                                        )}
                                                    />
                                                    <PageSeparator />
                                                    <FormField
                                                        control={form.control}
                                                        name="auth_server_data"
                                                        render={({ field }) => (
                                                            <FormItem className="flex items-center justify-between flex-wrap gap-4 mb-4">
                                                                <FormLabel>
                                                                    {authType === "LDAP"
                                                                        ? t("User's Full DN")
                                                                        : t("AD Username")}
                                                                </FormLabel>
                                                                <div className="w-md">
                                                                    <FormControl>
                                                                        <Input
                                                                            {...field}
                                                                            placeholder={
                                                                                authType === "LDAP"
                                                                                    ? "cn=John Smith,dn=nagios,dc=com"
                                                                                    : t("AD Username")
                                                                            }
                                                                        />
                                                                    </FormControl>
                                                                    <FormMessage />
                                                                </div>
                                                            </FormItem>
                                                        )}
                                                    />
                                                </>
                                            )}
                                        </CardContent>
                                    </Card>
                                    <Card
                                        ref={(el) => { sectionRefs.current["security"] = el }}
                                        id="security"
                                    >
                                        <CardContent>
                                            <div className="text-lg font-semibold mb-6 flex items-center gap-2">{<Shield />}{t("Security")}</div>
                                            <FormField
                                                control={form.control}
                                                name="role_id"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between gap-4 mb-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('Role')}</FormLabel>
                                                            <FormDescription>
                                                                {t("Assign a role to this user. Admins have full access. The Admin and User roles are protected and User is the default role assigned to users. ")}
                                                                <Link className="text-primary hover:underline" to="/admin/roles">{t("Manage roles here")}</Link>.
                                                            </FormDescription>
                                                        </div>
                                                        <FormControl>
                                                            <Select
                                                                value={`${field.value}`}
                                                                onValueChange={(value) => field.onChange(Number(value))}
                                                            >
                                                                <SelectTrigger className="w-xs">
                                                                    <SelectValue placeholder={t("Select a role")} />
                                                                </SelectTrigger>
                                                                <SelectContent>
                                                                    <SelectGroup>
                                                                        {roles.map((role) => (
                                                                            <SelectItem key={role.id} value={`${role.id}`}>
                                                                                {role.type === 'admin' && <Shield />}
                                                                                {role.type === 'user' && <UserIcon />}
                                                                                {role.type === 'custom' && <Settings />}
                                                                                {role.name}
                                                                            </SelectItem>
                                                                        ))}
                                                                    </SelectGroup>
                                                                </SelectContent>
                                                            </Select>
                                                        </FormControl>
                                                        <FormMessage />
                                                    </FormItem>
                                                )}
                                            />
                                            <PageSeparator />
                                            <FormField
                                                control={form.control}
                                                name="apiaccess"
                                                render={({ field }) => (
                                                    <FormItem className="flex items-center justify-between flex-wrap gap-4">
                                                        <div className="space-y-1">
                                                            <FormLabel>{t('API Access')}</FormLabel>
                                                            <FormDescription>{t('If you want to allow this user to use the external API via an access key')}</FormDescription>
                                                        </div>
                                                        <FormControl className="w-md">
                                                            <RadioGroup
                                                                className="gap-4 grid"
                                                                onValueChange={field.onChange}
                                                                value={field.value}
                                                            >
                                                                <div className="flex items-center space-x-2">
                                                                    <RadioGroupItem value="1" id="api-option-one" />
                                                                    <Label htmlFor="api-option-one">{t('Yes')}</Label>
                                                                </div>
                                                                <div className="flex items-center space-x-2">
                                                                    <RadioGroupItem value="0" id="api-option-two" />
                                                                    <Label htmlFor="api-option-two">{t('No')}</Label>
                                                                </div>
                                                            </RadioGroup>
                                                        </FormControl>
                                                        <FormMessage />
                                                    </FormItem>
                                                )}
                                            />
                                        </CardContent>
                                    </Card>
                                </ScrollingContainer>
                                <DialogFooter className="h-16 p-3.5">
                                    <Button
                                        type="button"
                                        variant="secondary"
                                        onClick={() => handleDialogChange(false)}
                                    >
                                        {t("Cancel")}
                                    </Button>
                                    <Button type="submit">
                                        {user ? t('Update User') : t('Create User')}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </Form>
                    </div>
                </SidebarProvider>
            </DialogContent>
        </Dialog>
    );
}