import { useTranslation } from "react-i18next";
import { toast } from "sonner";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import axios, { AxiosError } from "@/lib/axios";
import { PersonalInformation } from "@/components/user/profile/types";

export function useUpdatePersonalInformation() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ data }: { data: PersonalInformation }) => {
      return axios.patch(
        "/api/v1/profile/personal-information",
        data,
        {
          headers: {
              "content-type": "application/x-www-form-urlencoded",
          },
        }
      )
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["auth"] });
      toast(t("Profile information updated successfully."));
    },
    onError: () => {
      toast(t("Failed to update profile information."), {
        description: t("Please try again."),
      });
    },
  })
}

export function useUpdateLanguage() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (lang: string) => {
      return axios.patch(
        "/api/v1/profile/language",
        {
          lang
        },
        {
          headers: {
              "content-type": "application/x-www-form-urlencoded",
          },
        }
      )
    },
    onSuccess: (response) => {
      const { language: lng } = response.data;
      queryClient.invalidateQueries({ queryKey: ["auth"] });
      localStorage.removeItem('login_locale');
      queryClient.invalidateQueries({ queryKey: ["language"] });
      toast(t("Language updated successfully.", { lng }));
    },
    onError: () => {
      toast(t("Failed to update language."), {
        description: t("Please try again."),
      });
    },
  })
}

export function useUpdateTheme() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (theme: string) => {
      return axios.patch(
        "/api/v1/profile/theme",
        {
          theme
        },
        {
          headers: {
              "content-type": "application/x-www-form-urlencoded",
          },
        }
      )
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["auth"] });
      queryClient.invalidateQueries({ queryKey: ["theme"] });
      toast(t("Theme updated successfully."));
    },
    onError: () => {
      toast(t("Failed to update theme."), {
        description: t("Please try again."),
      });
    },
  })
}

export function useUpdatePassword(setOpen: (open: boolean) => void) {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({
      current_password,
      new_password,
      confirm_password,
    }: {
      current_password: string;
      new_password: string;
      confirm_password: string;
    }) => {
      return axios.patch(
        "/api/v1/profile/password",
        {
          current_password,
          new_password,
          confirm_password,
        },
        {
          headers: {
            "content-type": "application/x-www-form-urlencoded",
          },
        }
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["auth"] });
      setOpen(false);
      toast(t("Password updated successfully."));
    },
    onError: (error: AxiosError<{ message: string }>) => {
      toast(t("Failed to edit password."), {
        description: error.response?.data.message ?? t("Please try again."),
      });
    },
  });
}

export function useUpdateAPIKey() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (apikey_id: number) => {
      return axios.patch(
        "/api/v1/profile/api-key",
        `apikey_id=${apikey_id}`,
        {
          headers: {
            "content-type": "application/x-www-form-urlencoded",
          },
        }
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["auth"] });
      toast(t("API key generated successfully."));
    },
    onError: () => {
      toast(t("Failed to generate API key."), {
        description: t("Please try again."),
      });
    },
  });
}