import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import axios, { AxiosError } from '@/lib/axios';
import { useTranslation } from 'react-i18next';
import { toast } from 'sonner';
import {  WiresharkCapture, } from '@/components/wireshark/capturehistory/types';

export function useGetWiresharkCaptures() {
    return useQuery<WiresharkCapture[], AxiosError>({
        queryKey: ['wiresharkCapturesHistory'],
        queryFn: async () => {
        const response = await axios.get('/api/v1/wireshark/captures');
        return response.data.captures;
        },
    });
}

export function useDeleteCaptures() {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (captureIds: number[]) =>
      axios.delete(`/api/v1/wireshark/captures`, {
        data: { ids: captureIds },
      }),

    onMutate: async (captureIds) => {
      const key = ['wiresharkCapturesHistory'];
      await queryClient.cancelQueries({ queryKey: key });

      const previousCaptures = queryClient.getQueryData<WiresharkCapture[]>(key);

      queryClient.setQueryData(key, (old: WiresharkCapture[] = []) =>
        old.filter(capture => !captureIds.includes(capture.id))
      );

      return { previousCaptures };
    },

    onSuccess: (_data, captureIds) => {
      const message = captureIds.length > 1
        ? t('Captures deleted successfully.')
        : t('Capture deleted successfully.');
      toast.success(message);
    },

    onError: (_error, captureIds, context) => {
      queryClient.setQueryData(['wiresharkCapturesHistory'], context?.previousCaptures);
      const errorMessage = captureIds.length > 1
        ? t('Failed to delete captures.')
        : t('Failed to delete capture.');
      toast.error(errorMessage, {
        description: t('Please try again.'),
      });
    },

    onSettled: () => {
      queryClient.invalidateQueries({ queryKey: ['wiresharkCapturesHistory'] });
      queryClient.refetchQueries({ queryKey: ['wiresharkCapturesHistory'] });
    },
  });
}

export function useDownloadCapture() {
  return useMutation({
    mutationFn: async (captureId: number) => {
      const response = await axios.get(`/api/v1/wireshark/captures/${captureId}/download`, {
        responseType: 'blob',
      });
      return { blob: response.data, captureId };
    },
  });
}

export function useScanWithSuricata() {
  return useMutation({
    mutationFn: async (captureId: number) => {
      const response = await axios.post(
        `/api/v1/wireshark/captures/${captureId}/scan-with-suricata`
      );
      toast.info("Scanning capture PCAP with Suricata");
      return response.data;
    },
  });
}

export function useImportCapture(options?: { onSuccess?: () => void }) {
  const { t } = useTranslation();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (formData: FormData) => {
      const response = await axios.post(`/api/v1/wireshark/captures/import`, formData);
      return response.data;
    },
    onSuccess: () => {
      toast.success(t('PCAP file uploaded successfully.'));
      queryClient.invalidateQueries({ queryKey: ['wiresharkCapturesHistory'] }); // Refresh history
      options?.onSuccess?.(); // Close dialog, reset file, etc.
    },
    onError: (error: AxiosError) => {
      toast.error(t('Failed to upload PCAP file'), {
        description: error.message,
      });
    },
  });
}
