import { Pie, PieChart, Cell } from "recharts"
import { useTranslation } from 'react-i18next';
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import {
  ChartConfig,
  ChartContainer,
  ChartTooltip,
  ChartTooltipContent,
  ChartLegend,
  ChartLegendContent,
} from "@/components/ui/chart"

interface ProtocolPieChartProps {
  protocolUsage: {
    protocol: string
    packet_count: number
  }[]
}

const chartConfig = {
  packet_count: {
    label: "Packet Count",
  },
} satisfies ChartConfig

export function ProtocolPieChart({ protocolUsage }: ProtocolPieChartProps) {
  const { t } = useTranslation();
  const protocolData = protocolUsage.map((item, index) => ({
    ...item,
    fill: `var(--chart-${(index % 10) + 1})`,
  }));

  const dynamicChartConfig: ChartConfig = {
    ...chartConfig,
    ...protocolData.reduce((acc, item) => {
      acc[item.protocol] = {
        label: item.protocol,
        color: item.fill,
      };
      return acc;
    }, {} as ChartConfig),
  };

  return (
    <Card className="flex flex-col">
      <CardHeader className="items-center pb-0">
        <CardTitle>{t('Protocol Usage')}</CardTitle>
        <CardDescription>{t('Protocol usage by packet count')}</CardDescription>
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        <ChartContainer
          config={dynamicChartConfig}
          className="mx-auto aspect-square max-h-[300px]"
        >
          <PieChart>
            <ChartTooltip
              cursor={false}
              content={<ChartTooltipContent hideLabel />}
            />
            <Pie
              data={protocolData}
              dataKey="packet_count"
              nameKey="protocol"
              outerRadius="100%"
              isAnimationActive={false}
            >
              {protocolData.map((entry, index) => (
                <Cell key={`cell-${index}`} fill={entry.fill} />
              ))}
            </Pie>
            <ChartLegend
              content={<ChartLegendContent nameKey="protocol" />}
              className="-translate-y-2 flex-wrap gap-1 *:basis-1/9 *:justify-center"
            />
          </PieChart>
        </ChartContainer>
      </CardContent>
      <CardFooter className="flex-col gap-2 text-sm" />
    </Card>
  )
}
