import { Pie, PieChart, Cell} from "recharts"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { ChartConfig, ChartContainer, ChartTooltip, ChartTooltipContent, ChartLegend, ChartLegendContent } from "@/components/ui/chart"
import { cn } from "@/lib/utils"
import { useTranslation } from "react-i18next"

interface PieChartProps {
  title: string
  description?: string
  data: Record<string, number>
  className?: string
}

export function OverviewPieChart({ title, description, data, className }: PieChartProps) {
  const { t } = useTranslation();

  const chartData = Object.entries(data)
    .filter(([_, value]) => value > 0)
    .map(([label, value], index) => ({
      label,
      value,
      fill: `var(--chart-${(index % 10) + 1})`,
    }))

  const chartConfig: ChartConfig = {
    value: { label: "Value" },
    ...Object.fromEntries(
      chartData.map(item => [item.label, { label: item.label, color: item.fill }])
    ),
  } satisfies ChartConfig

  const hasData = chartData.length > 0

  return (
    <Card className={cn("flex flex-col", className)}>
      <CardHeader className="items-center pb-0">
        <CardTitle>{t(title)}</CardTitle>
        {description && <CardDescription>{t(description)}</CardDescription>}
      </CardHeader>
      <CardContent className="flex-1 pb-0">
        {hasData ? (
          <ChartContainer config={chartConfig} className="mx-auto aspect-square max-h-[250px]">
            <PieChart>
              <ChartTooltip cursor={false} content={<ChartTooltipContent hideLabel />} />
              <Pie data={chartData} dataKey="value" nameKey="label">
                {chartData.map(item => (
                  <Cell key={item.label} fill={item.fill} />
                ))}
              </Pie>
              <ChartLegend
                content={<ChartLegendContent nameKey="label" />}
                className="-translate-y-2 flex-wrap gap-2 *:basis-1 *:justify-center"
              />
            </PieChart>
          </ChartContainer>
        ) : (
          <div className="mx-auto aspect-square max-h-[250px] flex items-center justify-center text-muted-foreground">
            {t("No data available")}
          </div>
        )}
      </CardContent>
    </Card>
  )
}
