import * as React from "react";
import { z } from "zod";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Dialog,
  DialogClose,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Tooltip, TooltipTrigger, TooltipContent } from "@/components/ui/tooltip";
import { useTranslation } from "react-i18next";
import { Plus, FilePenLine } from "lucide-react";
import { useGetNetworkInterfaces } from "@/components/wireshark/livecapture/queries";
import { NetworkInterface } from "@/components/wireshark/livecapture/types";
import { useCreateRingBuffer, useEditRingBuffer } from "@/components/wireshark/ringbuffer/queries";
import { MultiSelect } from "@/components/ui/multi-select";
import { RingBuffer } from "@/components/wireshark/ringbuffer/types"; 
import { useQueryClient } from "@tanstack/react-query";

const FormSchema = z.object({
  interfaces: z.array(z.string()).min(1, "Select at least one interface"),
  number_of_files: z.coerce.number<number>().min(1, "Must be at least 1"),
  file_size: z.coerce.number<number>().min(1, "Must be at least 1 KB"),
  directory: z.string().min(1, "Directory is required"),
  file_name: z.string().min(1, "File name is required"),
});

type FormValues = z.infer<typeof FormSchema>;

export default function AddEditRingBufferDialog({ ringbuffer }: { ringbuffer?: RingBuffer }) {
  const { t } = useTranslation();
  const { data: interfacesData } = useGetNetworkInterfaces();
  const interfaces: NetworkInterface[] = interfacesData || [];
  const createMutation = useCreateRingBuffer();
  const editMutation = useEditRingBuffer();
  const [open, setOpen] = React.useState(false);
  const queryClient = useQueryClient();

  const form = useForm<FormValues>({
    resolver: zodResolver(FormSchema),
    mode: "onChange",
    reValidateMode: "onChange",
    defaultValues: ringbuffer ?? {
      interfaces: ["any"],
      number_of_files: 5,
      file_size: 1024,
      directory: "/usr/local/wireshark/ringbuffer",
      file_name: "capture",
    },
  });

  const onSubmit = (data: FormValues) => {
    if (ringbuffer) {
      editMutation.mutate(
        { ...data, id: ringbuffer.id },
        {
          onSuccess: () => {
            setOpen(false);
            queryClient.invalidateQueries({ queryKey: ['ringbuffer'] });
          }
        }
      );
    } else {
      createMutation.mutate(data);
    }
  };
  return (
    <Dialog open={open} onOpenChange={setOpen}>
      {ringbuffer ? (
        <Tooltip>
          <TooltipTrigger asChild>
            <DialogTrigger asChild>
              <Button>
                <FilePenLine />
              </Button>
            </DialogTrigger>
          </TooltipTrigger>
          <TooltipContent>
            <p>{t("Edit Ring Buffer config")}</p>
          </TooltipContent>
        </Tooltip>
      ) : (
        <DialogTrigger asChild>
          <Button>
            <Plus />
            {t("Create a Ring Buffer")}
          </Button>
        </DialogTrigger>
      )}
      <DialogContent className="sm:max-w-[500px]">
        <DialogHeader>
          <DialogTitle>
            {ringbuffer ? t("Edit Ring Buffer") : t("Create A Ring Buffer")}
          </DialogTitle>
          <DialogDescription>
            {t("Fill in the details to start the ring buffer capture.")}
          </DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
            <FormField
              control={form.control}
              name="interfaces"
              render={({ field }) => (
                <FormItem className="space-y-2">
                  <FormLabel>Network Interfaces</FormLabel>
                  <FormControl>
                    <MultiSelect
                      options={interfaces.map((iface) => ({
                        label: iface.name,
                        value: iface.name,
                      }))}
                      onValueChange={(selected) =>
                        form.setValue("interfaces", selected, {
                          shouldDirty: true,
                          shouldTouch: true,
                          shouldValidate: true,
                        })
                      }
                      defaultValue={field.value || []}
                      placeholder="Select Interfaces"
                      maxCount={8} 
                      useSearch={true} 
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="number_of_files"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Number of Files</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="file_size"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>File Size (KB)</FormLabel>
                  <FormControl>
                    <Input {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="directory"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Directory</FormLabel>
                  <FormControl>
                    <Input
                      placeholder="/usr/local/wireshark/ringbuffer"
                      {...field}
                      readOnly
                      disabled
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="file_name"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>File Name</FormLabel>
                  <FormControl>
                    <Input placeholder="capture" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <DialogFooter className="space-x-2">
              <DialogClose asChild>
                <Button variant="outline">{t("Cancel")}</Button>
              </DialogClose>
              <Button type="submit">
                {ringbuffer ? t("Update") : t("Create")}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}