import axios from "@/lib/axios"
import { createContext, useContext } from "react"
import { User } from "@/components/user/types";
import { useQuery } from "@tanstack/react-query";
import { License } from "@/components/administration/license/types";
import { useEnvironment } from "@/contexts/EnvironmentContext";

type AuthProviderState = {
  user: User | null;
  installed: boolean;
  isUserLoading: boolean;
  isInstalledLoading: boolean;
  license: License | null;
  isLicenseLoading: boolean;
}

const initialState: AuthProviderState = {
  user: null,
  installed: false,
  isUserLoading: true,
  isInstalledLoading: true,
  license: null,
  isLicenseLoading: true,
}

const AuthProviderContext = createContext<AuthProviderState>(initialState)

export function AuthProvider({
  children,
  ...props
}: { children: React.ReactNode }) {
  const { environment } = useEnvironment();

  const { data: installed, isLoading: isInstalledLoading } = useQuery({
    queryKey: ['installed'],
    queryFn: async () => {
      const response = await axios.get('/api/v1/auth/installed');
      return response.data.value == 1;
    },
    retry: false,
    refetchOnWindowFocus: false,
  })
 
  const { data: user, isLoading: isUserLoading } = useQuery({
    queryKey: ['auth'],
    queryFn: async () => {
      try {
        const response = await axios.get('/api/v1/auth');
        return response.data;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchOnWindowFocus: false,
    refetchInterval: ((environment?.sessionLifetime ?? 0) * 60 * 1000) + 5000, // Refetch every session lifetime + a buffer of 5 seconds
  })

  const { data: license, isLoading: isLicenseLoading } = useQuery<License | null>({
    queryKey: ["license"],
    queryFn: async () => {
      try {
        const response = await axios.get("/api/v1/license");
        return response.data as License;
      } catch {
        return null;
      }
    },
    retry: false,
    refetchOnWindowFocus: false,
    refetchInterval: (query) => {
      const license = query.state.data as License | null;

      if (license?.trial_seconds_left && license.trial_seconds_left > 0) {
        const interval = license.trial_seconds_left * 1000;

        // Cap the interval to 24 days to prevent JavaScript timer overflow.
        return Math.min(interval, 24 * 24 * 60 * 60 * 1000);
      }

      return false;
    },
  })

  return (
    <AuthProviderContext.Provider 
      {...props} 
      value={{ 
        user, 
        installed: installed ?? false, 
        isUserLoading, 
        isInstalledLoading,
        license: license ?? null,
        isLicenseLoading,
      }}
    >
      {children}
    </AuthProviderContext.Provider>
  )
}

export const useAuth = () => {
  const context = useContext(AuthProviderContext)

  if (context === undefined)
    throw new Error("useAuth must be used within an AuthProvider")

  return context
}
