import { createContext, useContext } from "react"
import { useQuery } from "@tanstack/react-query";
import axios from "@/lib/axios";
type Theme = "dark" | "light" | "default"

type ThemeProviderProps = {
  children: React.ReactNode
  defaultTheme?: Theme
  storageKey?: string
}

const initialState: Theme = 'dark';

const ThemeProviderContext = createContext<Theme>(initialState)

export function ThemeProvider({
  children,
  defaultTheme = "dark",
  storageKey = "vite-ui-theme",
  ...props
}: ThemeProviderProps) {
  const themeQuery = useQuery({
    queryKey: ['theme'],
    queryFn: async () => {
      try {
        const response = await axios.get("/api/v1/system/theme");
        const theme: Theme = response.data.theme ?? "dark";

        const root = window.document.documentElement;
        root.classList.remove("light", "dark");
        root.classList.add(theme);

        return theme;
      } catch (error) {
        const root = window.document.documentElement;
        root.classList.remove("light", "dark");
        root.classList.add("dark");

        return "dark";
      }
    },
  });

  if (!themeQuery.data) return;

  return (
    <ThemeProviderContext.Provider {...props} value={themeQuery.data}>
      {children}
    </ThemeProviderContext.Provider>
  )
}

export const useTheme = () => {
  const context = useContext(ThemeProviderContext)

  if (context === undefined)
    throw new Error("useTheme must be used within a ThemeProvider")

  return context
}
